/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.scheduler;

import java.util.ArrayList;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.neo4j.kernel.impl.scheduler.ThreadPoolManager;
import org.neo4j.kernel.impl.scheduler.TimeBasedTaskScheduler;
import org.neo4j.scheduler.Group;
import org.neo4j.scheduler.JobHandle;
import org.neo4j.time.FakeClock;
import org.neo4j.time.SystemNanoClock;
import org.neo4j.util.concurrent.BinaryLatch;

public class TimeBasedTaskSchedulerTest {
    private FakeClock clock;
    private ThreadPoolManager pools;
    private TimeBasedTaskScheduler scheduler;
    private AtomicInteger counter;
    private Semaphore semaphore;

    @Before
    public void setUp() {
        this.clock = new FakeClock();
        this.pools = new ThreadPoolManager(new ThreadGroup("TestPool"));
        this.scheduler = new TimeBasedTaskScheduler((SystemNanoClock)this.clock, this.pools);
        this.counter = new AtomicInteger();
        this.semaphore = new Semaphore(0);
    }

    @After
    public void tearDown() {
        InterruptedException exception = this.pools.shutDownAll();
        if (exception != null) {
            throw new RuntimeException("Test was interrupted?", exception);
        }
    }

    private void assertSemaphoreAcquire() throws InterruptedException {
        long timeoutMillis = TimeUnit.SECONDS.toMillis(10L);
        long sleepIntervalMillis = 10L;
        long iterations = timeoutMillis / sleepIntervalMillis;
        int i = 0;
        while ((long)i < iterations) {
            if (this.semaphore.tryAcquire(sleepIntervalMillis, TimeUnit.MILLISECONDS)) {
                return;
            }
            this.scheduler.tick();
            ++i;
        }
        Assert.fail((String)"Semaphore acquire timeout");
    }

    @Test
    public void mustDelayExecution() throws Exception {
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, this.counter::incrementAndGet, 100L, 0L);
        this.scheduler.tick();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)0));
        this.clock.forward(99L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)0));
        this.clock.forward(1L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle.waitTermination();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void mustOnlyScheduleTasksThatAreDue() throws Exception {
        JobHandle handle1 = this.scheduler.submit(Group.STORAGE_MAINTENANCE, () -> this.counter.addAndGet(10), 100L, 0L);
        JobHandle handle2 = this.scheduler.submit(Group.STORAGE_MAINTENANCE, () -> this.counter.addAndGet(100), 200L, 0L);
        this.scheduler.tick();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)0));
        this.clock.forward(199L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle1.waitTermination();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)10));
        this.clock.forward(1L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle2.waitTermination();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)110));
    }

    @Test
    public void mustNotRescheduleDelayedTasks() throws Exception {
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, this.counter::incrementAndGet, 100L, 0L);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle.waitTermination();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)1));
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle.waitTermination();
        this.pools.getThreadPool(Group.STORAGE_MAINTENANCE).shutDown();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void mustRescheduleRecurringTasks() throws Exception {
        this.scheduler.submit(Group.STORAGE_MAINTENANCE, this.semaphore::release, 100L, 100L);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.assertSemaphoreAcquire();
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.assertSemaphoreAcquire();
    }

    @Test
    public void mustNotRescheduleRecurringTasksThatThrows() throws Exception {
        Runnable runnable = () -> {
            this.semaphore.release();
            throw new RuntimeException("boom");
        };
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, runnable, 100L, 100L);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.assertSemaphoreAcquire();
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        try {
            handle.waitTermination();
            Assert.fail((String)"waitTermination should have thrown because the task should have failed.");
        }
        catch (ExecutionException e) {
            Assert.assertThat((Object)e.getCause().getMessage(), (Matcher)Matchers.is((Object)"boom"));
        }
        Assert.assertThat((Object)this.semaphore.drainPermits(), (Matcher)Matchers.is((Object)0));
    }

    @Test
    public void mustNotStartRecurringTasksWherePriorExecutionHasNotYetFinished() {
        Runnable runnable = () -> {
            this.counter.incrementAndGet();
            this.semaphore.acquireUninterruptibly();
        };
        this.scheduler.submit(Group.STORAGE_MAINTENANCE, runnable, 100L, 100L);
        for (int i = 0; i < 4; ++i) {
            this.scheduler.tick();
            this.clock.forward(100L, TimeUnit.NANOSECONDS);
        }
        this.semaphore.release(Integer.MAX_VALUE);
        this.pools.getThreadPool(Group.STORAGE_MAINTENANCE).shutDown();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void longRunningTasksMustNotDelayExecutionOfOtherTasks() throws Exception {
        BinaryLatch latch = new BinaryLatch();
        Runnable longRunning = () -> ((BinaryLatch)latch).await();
        Runnable shortRunning = this.semaphore::release;
        this.scheduler.submit(Group.STORAGE_MAINTENANCE, longRunning, 100L, 100L);
        this.scheduler.submit(Group.STORAGE_MAINTENANCE, shortRunning, 100L, 100L);
        for (int i = 0; i < 4; ++i) {
            this.clock.forward(100L, TimeUnit.NANOSECONDS);
            this.scheduler.tick();
            this.assertSemaphoreAcquire();
        }
        latch.release();
    }

    @Test
    public void delayedTasksMustNotRunIfCancelledFirst() throws Exception {
        ArrayList cancelListener = new ArrayList();
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, this.counter::incrementAndGet, 100L, 0L);
        handle.registerCancelListener(cancelListener::add);
        this.clock.forward(90L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        handle.cancel(false);
        this.clock.forward(10L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.pools.getThreadPool(Group.STORAGE_MAINTENANCE).shutDown();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)0));
        Assert.assertThat(cancelListener, (Matcher)Matchers.contains((Object[])new Boolean[]{Boolean.FALSE}));
        try {
            handle.waitTermination();
            Assert.fail((String)"waitTermination should have thrown a CancellationException.");
        }
        catch (CancellationException cancellationException) {
            // empty catch block
        }
    }

    @Test
    public void recurringTasksMustStopWhenCancelled() throws InterruptedException {
        ArrayList cancelListener = new ArrayList();
        Runnable recurring = () -> {
            this.counter.incrementAndGet();
            this.semaphore.release();
        };
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, recurring, 100L, 100L);
        handle.registerCancelListener(cancelListener::add);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.assertSemaphoreAcquire();
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.assertSemaphoreAcquire();
        handle.cancel(true);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.pools.getThreadPool(Group.STORAGE_MAINTENANCE).shutDown();
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)2));
        Assert.assertThat(cancelListener, (Matcher)Matchers.contains((Object[])new Boolean[]{Boolean.TRUE}));
    }

    @Test
    public void overdueRecurringTasksMustStartAsSoonAsPossible() {
        Runnable recurring = () -> {
            this.counter.incrementAndGet();
            this.semaphore.acquireUninterruptibly();
        };
        JobHandle handle = this.scheduler.submit(Group.STORAGE_MAINTENANCE, recurring, 100L, 100L);
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        while (this.counter.get() < 1) {
            Thread.yield();
        }
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.scheduler.tick();
        this.clock.forward(100L, TimeUnit.NANOSECONDS);
        this.semaphore.release();
        this.scheduler.tick();
        long deadline = System.nanoTime() + TimeUnit.SECONDS.toNanos(10L);
        while (this.counter.get() < 2 && System.nanoTime() < deadline) {
            this.scheduler.tick();
            Thread.yield();
        }
        Assert.assertThat((Object)this.counter.get(), (Matcher)Matchers.is((Object)2));
        this.semaphore.release(Integer.MAX_VALUE);
        handle.cancel(false);
    }
}

