/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log.files;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.kernel.impl.api.TestCommandReaderFactory;
import org.neo4j.kernel.impl.transaction.SimpleLogVersionRepository;
import org.neo4j.kernel.impl.transaction.SimpleTransactionIdStore;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryReader;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeader;
import org.neo4j.kernel.impl.transaction.log.entry.VersionAwareLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.storageengine.api.CommandReaderFactory;
import org.neo4j.storageengine.api.LogVersionRepository;
import org.neo4j.storageengine.api.StoreId;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;

@Neo4jLayoutExtension
class TransactionLogFilesTest {
    @Inject
    private FileSystemAbstraction fileSystem;
    @Inject
    private DatabaseLayout databaseLayout;
    private final String filename = "filename";

    TransactionLogFilesTest() {
    }

    @Test
    void shouldGetTheFileNameForAGivenVersion() throws IOException {
        LogFiles files = this.createLogFiles();
        int version = 12;
        File versionFileName = files.getLogFileForVersion(12L);
        File expected = this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName(12));
        Assertions.assertEquals((Object)expected, (Object)versionFileName);
    }

    @Test
    void extractHeaderOf3_5Format() throws IOException {
        LogFiles files = this.createLogFiles();
        this.create3_5FileWithHeader(this.databaseLayout, "0");
        this.create3_5FileWithHeader(this.databaseLayout, "1", 1);
        this.create3_5FileWithHeader(this.databaseLayout, "2", 2);
        LogHeader logHeader = files.extractHeader(0L);
        Assertions.assertEquals((long)16L, (long)logHeader.getStartPosition().getByteOffset());
        Assertions.assertEquals((byte)6, (byte)logHeader.getLogFormatVersion());
        Assertions.assertEquals((long)16L, (long)files.extractHeader(1L).getStartPosition().getByteOffset());
        Assertions.assertEquals((long)16L, (long)files.extractHeader(2L).getStartPosition().getByteOffset());
    }

    @Test
    void detectEntriesIn3_5Format() throws IOException {
        LogFiles files = this.createLogFiles();
        this.create3_5FileWithHeader(this.databaseLayout, "0");
        this.create3_5FileWithHeader(this.databaseLayout, "1", 10);
        Assertions.assertFalse((boolean)files.hasAnyEntries(0L));
        Assertions.assertTrue((boolean)files.hasAnyEntries(1L));
    }

    @Test
    void shouldVisitEachLofFile() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("1"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("some", "2"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("3"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, "filename")).close();
        ArrayList seenFiles = new ArrayList();
        ArrayList seenVersions = new ArrayList();
        files.accept((file, logVersion) -> {
            seenFiles.add(file);
            seenVersions.add(logVersion);
        });
        MatcherAssert.assertThat(seenFiles, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("filename", "1")), this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("filename", "3"))}));
        MatcherAssert.assertThat(seenVersions, (Matcher)Matchers.containsInAnyOrder((Object[])new Long[]{1L, 3L}));
        files.shutdown();
    }

    @Test
    void shouldBeAbleToRetrieveTheHighestLogVersion() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("1"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("some", "4"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, this.getVersionedLogFileName("3"))).close();
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, "filename")).close();
        long highestLogVersion = files.getHighestLogVersion();
        Assertions.assertEquals((long)3L, (long)highestLogVersion);
        files.shutdown();
    }

    @Test
    void shouldReturnANegativeValueIfThereAreNoLogFiles() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(this.databaseLayout.file(this.getVersionedLogFileName("some", "4"))).close();
        this.fileSystem.write(this.databaseLayout.file("filename")).close();
        long highestLogVersion = files.getHighestLogVersion();
        Assertions.assertEquals((long)-1L, (long)highestLogVersion);
        files.shutdown();
    }

    @Test
    void shouldFindTheVersionBasedOnTheFilename() throws Throwable {
        LogFiles logFiles = this.createLogFiles();
        File file = new File("v....2");
        long logVersion = logFiles.getLogVersion(file);
        Assertions.assertEquals((long)2L, (long)logVersion);
        logFiles.shutdown();
    }

    @Test
    void shouldThrowIfThereIsNoVersionInTheFileName() throws IOException {
        LogFiles logFiles = this.createLogFiles();
        File file = new File("wrong");
        RuntimeException exception = (RuntimeException)Assertions.assertThrows(RuntimeException.class, () -> logFiles.getLogVersion(file));
        Assertions.assertEquals((Object)("Invalid log file '" + file.getName() + "'"), (Object)exception.getMessage());
    }

    @Test
    void shouldThrowIfVersionIsNotANumber() throws IOException {
        LogFiles logFiles = this.createLogFiles();
        File file = new File(this.getVersionedLogFileName("aa", "A"));
        Assertions.assertThrows(NumberFormatException.class, () -> logFiles.getLogVersion(file));
    }

    @Test
    void isLogFile() throws IOException {
        LogFiles logFiles = this.createLogFiles();
        Assertions.assertFalse((boolean)logFiles.isLogFile(new File("aaa.tx.log")));
        Assertions.assertTrue((boolean)logFiles.isLogFile(new File("filename.0")));
        Assertions.assertTrue((boolean)logFiles.isLogFile(new File("filename.17")));
    }

    @Test
    void emptyFileWithoutEntriesDoesNotHaveThem() throws IOException {
        LogFiles logFiles = this.createLogFiles();
        String file = this.getVersionedLogFileName("1");
        this.fileSystem.write(this.createTransactionLogFile(this.databaseLayout, file)).close();
        Assertions.assertFalse((boolean)logFiles.hasAnyEntries(1L));
    }

    @Test
    void fileWithoutEntriesDoesNotHaveThemIndependentlyOfItsSize() throws IOException {
        LogFiles logFiles = this.createLogFiles();
        try (PhysicalLogVersionedStoreChannel channel = logFiles.createLogChannelForVersion(1L, () -> 1L);){
            MatcherAssert.assertThat((Object)channel.size(), (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Long.valueOf(64L)));
            Assertions.assertFalse((boolean)logFiles.hasAnyEntries(1L));
        }
    }

    private void create3_5FileWithHeader(DatabaseLayout databaseLayout, String version, int bytesOfData) throws IOException {
        try (StoreChannel storeChannel = this.fileSystem.write(this.createTransactionLogFile(databaseLayout, this.getVersionedLogFileName(version)));){
            ByteBuffer byteBuffer = ByteBuffers.allocate((int)(16 + bytesOfData));
            while (byteBuffer.hasRemaining()) {
                byteBuffer.put((byte)6);
            }
            byteBuffer.flip();
            storeChannel.writeAll(byteBuffer);
        }
    }

    private void create3_5FileWithHeader(DatabaseLayout databaseLayout, String version) throws IOException {
        this.create3_5FileWithHeader(databaseLayout, version, 0);
    }

    private File createTransactionLogFile(DatabaseLayout databaseLayout, String fileName) {
        File transactionLogsDirectory = databaseLayout.getTransactionLogsDirectory();
        return new File(transactionLogsDirectory, fileName);
    }

    private LogFiles createLogFiles() throws IOException {
        return LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem).withLogFileName("filename").withTransactionIdStore((TransactionIdStore)new SimpleTransactionIdStore()).withLogVersionRepository((LogVersionRepository)new SimpleLogVersionRepository()).withLogEntryReader((LogEntryReader)new VersionAwareLogEntryReader((CommandReaderFactory)new TestCommandReaderFactory())).withStoreId(StoreId.UNKNOWN).build();
    }

    private String getVersionedLogFileName(int version) {
        return this.getVersionedLogFileName("filename", String.valueOf(version));
    }

    private String getVersionedLogFileName(String version) {
        return this.getVersionedLogFileName("filename", version);
    }

    private String getVersionedLogFileName(String filename, String version) {
        return filename + "." + version;
    }
}

