/*
 * This file is part of dependency-check-core.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (c) 2012 Jeremy Long. All Rights Reserved.
 */
package org.owasp.dependencycheck.dependency;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.commons.lang3.builder.CompareToBuilder;

/**
 * Contains the information about a vulnerability.
 *
 * @author Jeremy Long
 */
@NotThreadSafe
public class Vulnerability implements Serializable, Comparable<Vulnerability> {

    /**
     * An enumeration for the source of vulnerability.
     */
    public enum Source {
        /**
         * National Vulnerability Database.
         */
        NVD,
        /**
         * Node Security Platform.
         */
        NSP
    }

    /**
     * The serial version uid.
     */
    private static final long serialVersionUID = 307319490326651052L;
    /**
     * The name of the vulnerability.
     */
    private String name;
    /**
     * the description of the vulnerability.
     */
    private String description;
    /**
     * References for this vulnerability.
     */
    private Set<Reference> references = new HashSet<>();
    /**
     * A set of vulnerable software.
     */
    private Set<VulnerableSoftware> vulnerableSoftware = new HashSet<>();
    /**
     * The CWE for the vulnerability.
     */
    private String cwe;
    /**
     * CVSS Score.
     */
    private float cvssScore;
    /**
     * CVSS Access Vector.
     */
    private String cvssAccessVector;
    /**
     * CVSS Access Complexity.
     */
    private String cvssAccessComplexity;

    /**
     * CVSS Authentication.
     */
    private String cvssAuthentication;
    /**
     * CVSS Confidentiality Impact.
     */
    private String cvssConfidentialityImpact;
    /**
     * CVSS Integrity Impact.
     */
    private String cvssIntegrityImpact;

    /**
     * CVSS Availability Impact.
     */
    private String cvssAvailabilityImpact;
    /**
     * The CPE id that caused this vulnerability to be flagged.
     */
    private String matchedCPE;
    /**
     * Whether or not all previous versions were affected.
     */
    private String matchedAllPreviousCPE;
    /**
     * Notes about the vulnerability. Generally used for suppression
     * information.
     */
    private String notes;

    /**
     * The source that identified the vulnerability.
     */
    private Source source = Source.NVD;

    /**
     * Get the value of name.
     *
     * @return the value of name
     */
    public String getName() {
        return name;
    }

    /**
     * Set the value of name.
     *
     * @param name new value of name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get the value of description.
     *
     * @return the value of description
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the value of description.
     *
     * @param description new value of description
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Get the value of references.
     *
     * @return the value of references
     */
    public Set<Reference> getReferences() {
        return references;
    }

    /**
     * Returns the list of references. This is primarily used within the
     * generated reports.
     *
     * @param sorted whether the returned list should be sorted
     * @return the list of references
     */
    public List<Reference> getReferences(boolean sorted) {
        final List<Reference> sortedRefs = new ArrayList<>(this.references);
        if (sorted) {
            Collections.sort(sortedRefs);
        }
        return sortedRefs;
    }

    /**
     * Set the value of references.
     *
     * @param references new value of references
     */
    public void setReferences(Set<Reference> references) {
        this.references = references;
    }

    /**
     * Adds a reference to the references collection.
     *
     * @param ref a reference for the vulnerability
     */
    public void addReference(Reference ref) {
        this.references.add(ref);
    }

    /**
     * Adds a reference.
     *
     * @param referenceSource the source of the reference
     * @param referenceName the referenceName of the reference
     * @param referenceUrl the url of the reference
     */
    public void addReference(String referenceSource, String referenceName, String referenceUrl) {
        final Reference ref = new Reference();
        ref.setSource(referenceSource);
        ref.setName(referenceName);
        ref.setUrl(referenceUrl);
        this.references.add(ref);
    }

    /**
     * Get the value of vulnerableSoftware.
     *
     * @return the value of vulnerableSoftware
     */
    public Set<VulnerableSoftware> getVulnerableSoftware() {
        return vulnerableSoftware;
    }

    /**
     * Returns a sorted list of vulnerable software. This is primarily used for
     * display within reports.
     *
     * @param sorted whether or not the list should be sorted
     * @return the list of vulnerable software
     */
    public List<VulnerableSoftware> getVulnerableSoftware(boolean sorted) {
        final List<VulnerableSoftware> sortedVulnerableSoftware = new ArrayList<>(this.vulnerableSoftware);
        if (sorted) {
            Collections.sort(sortedVulnerableSoftware);
        }
        return sortedVulnerableSoftware;
    }

    /**
     * Set the value of vulnerableSoftware.
     *
     * @param vulnerableSoftware new value of vulnerableSoftware
     */
    public void setVulnerableSoftware(Set<VulnerableSoftware> vulnerableSoftware) {
        this.vulnerableSoftware = vulnerableSoftware;
    }

    /**
     * Adds an entry for vulnerable software.
     *
     * @param cpe string representation of a CPE entry
     */
    public void addVulnerableSoftware(String cpe) {
        addVulnerableSoftware(cpe, null);
    }

    /**
     * Adds an entry for vulnerable software.
     *
     * @param cpe string representation of a cpe
     * @param previousVersion the previous version (previousVersion - cpe would
     * be considered vulnerable)
     */
    public void addVulnerableSoftware(String cpe, String previousVersion) {
        final VulnerableSoftware vs = new VulnerableSoftware();
        vs.setCpe(cpe);
        if (previousVersion != null) {
            vs.setPreviousVersion(previousVersion);
        }
        updateVulnerableSoftware(vs);
    }

    /**
     * Adds or updates a vulnerable software entry.
     *
     * @param vulnSoftware the vulnerable software
     */
    public void updateVulnerableSoftware(VulnerableSoftware vulnSoftware) {
        //this behavior is because the vuln software being updated may have
        // a value that is not included in the hash/comparison
        if (vulnerableSoftware.contains(vulnSoftware)) {
            vulnerableSoftware.remove(vulnSoftware);
        }
        vulnerableSoftware.add(vulnSoftware);
    }

    /**
     * Get the value of cwe.
     *
     * @return the value of cwe
     */
    public String getCwe() {
        return cwe;
    }

    /**
     * Set the value of cwe.
     *
     * @param cwe new value of cwe
     */
    public void setCwe(String cwe) {
        this.cwe = cwe;
    }

    /**
     * Get the value of cvssScore.
     *
     * @return the value of cvssScore
     */
    public float getCvssScore() {
        return cvssScore;
    }

    /**
     * Set the value of cvssScore.
     *
     * @param cvssScore new value of cvssScore
     */
    public void setCvssScore(float cvssScore) {
        this.cvssScore = cvssScore;
    }

    /**
     * Get the value of cvssAccessVector.
     *
     * @return the value of cvssAccessVector
     */
    public String getCvssAccessVector() {
        return cvssAccessVector;
    }

    /**
     * Set the value of cvssAccessVector.
     *
     * @param cvssAccessVector new value of cvssAccessVector
     */
    public void setCvssAccessVector(String cvssAccessVector) {
        this.cvssAccessVector = cvssAccessVector;
    }

    /**
     * Get the value of cvssAccessComplexity.
     *
     * @return the value of cvssAccessComplexity
     */
    public String getCvssAccessComplexity() {
        return cvssAccessComplexity;
    }

    /**
     * Set the value of cvssAccessComplexity.
     *
     * @param cvssAccessComplexity new value of cvssAccessComplexity
     */
    public void setCvssAccessComplexity(String cvssAccessComplexity) {
        this.cvssAccessComplexity = cvssAccessComplexity;
    }

    /**
     * Get the value of cvssAuthentication.
     *
     * @return the value of cvssAuthentication
     */
    public String getCvssAuthentication() {
        return cvssAuthentication;
    }

    /**
     * Set the value of cvssAuthentication.
     *
     * @param cvssAuthentication new value of cvssAuthentication
     */
    public void setCvssAuthentication(String cvssAuthentication) {
        this.cvssAuthentication = cvssAuthentication;
    }

    /**
     * Get the value of cvssConfidentialityImpact.
     *
     * @return the value of cvssConfidentialityImpact
     */
    public String getCvssConfidentialityImpact() {
        return cvssConfidentialityImpact;
    }

    /**
     * Set the value of cvssConfidentialityImpact.
     *
     * @param cvssConfidentialityImpact new value of cvssConfidentialityImpact
     */
    public void setCvssConfidentialityImpact(String cvssConfidentialityImpact) {
        this.cvssConfidentialityImpact = cvssConfidentialityImpact;
    }

    /**
     * Get the value of cvssIntegrityImpact.
     *
     * @return the value of cvssIntegrityImpact
     */
    public String getCvssIntegrityImpact() {
        return cvssIntegrityImpact;
    }

    /**
     * Set the value of cvssIntegrityImpact.
     *
     * @param cvssIntegrityImpact new value of cvssIntegrityImpact
     */
    public void setCvssIntegrityImpact(String cvssIntegrityImpact) {
        this.cvssIntegrityImpact = cvssIntegrityImpact;
    }

    /**
     * Get the value of cvssAvailabilityImpact.
     *
     * @return the value of cvssAvailabilityImpact
     */
    public String getCvssAvailabilityImpact() {
        return cvssAvailabilityImpact;
    }

    /**
     * Set the value of cvssAvailabilityImpact.
     *
     * @param cvssAvailabilityImpact new value of cvssAvailabilityImpact
     */
    public void setCvssAvailabilityImpact(String cvssAvailabilityImpact) {
        this.cvssAvailabilityImpact = cvssAvailabilityImpact;
    }

    /**
     * Get the value of notes from suppression notes.
     *
     * @return the value of notes
     */
    public String getNotes() {
        return notes;
    }

    /**
     * Set the value of notes.
     *
     * @param notes new value of cwe
     */
    public void setNotes(String notes) {
        this.notes = notes;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Vulnerability other = (Vulnerability) obj;
        return !((this.name == null) ? (other.name != null) : !this.name.equals(other.name));
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 41 * hash + (this.name != null ? this.name.hashCode() : 0);
        return hash;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("Vulnerability ");
        sb.append(this.name);
        sb.append("\nReferences:\n");
        for (Reference reference : getReferences(true)) {
            sb.append("=> ");
            sb.append(reference);
            sb.append("\n");
        }
        sb.append("\nSoftware:\n");

        for (VulnerableSoftware software : getVulnerableSoftware(true)) {
            sb.append("=> ");
            sb.append(software);
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Compares two vulnerabilities.
     *
     * @param v a vulnerability to be compared
     * @return a negative integer, zero, or a positive integer as this object is
     * less than, equal to, or greater than the specified vulnerability
     */
    @Override
    public int compareTo(Vulnerability v) {
        return new CompareToBuilder()
                .append(this.name, v.name)
                .toComparison();
    }

    /**
     * Sets the CPE that caused this vulnerability to be flagged.
     *
     * @param cpeId a CPE identifier
     * @param previous a flag indicating whether or not all previous versions
     * were affected (any non-null value is considered true)
     */
    public void setMatchedCPE(String cpeId, String previous) {
        matchedCPE = cpeId;
        matchedAllPreviousCPE = previous;
    }

    /**
     * Get the value of matchedCPE.
     *
     * @return the value of matchedCPE
     */
    public String getMatchedCPE() {
        return matchedCPE;
    }

    /**
     * Get the value of matchedAllPreviousCPE.
     *
     * @return the value of matchedAllPreviousCPE
     */
    public String getMatchedAllPreviousCPE() {
        return matchedAllPreviousCPE;
    }

    /**
     * Determines whether or not matchedAllPreviousCPE has been set.
     *
     * @return true if matchedAllPreviousCPE is not null; otherwise false
     */
    public boolean hasMatchedAllPreviousCPE() {
        return matchedAllPreviousCPE != null;
    }

    /**
     * Retruns the source that identified the vulnerability.
     *
     * @return the source
     */
    public Source getSource() {
        return source;
    }

    /**
     * Sets the source that identified the vulnerability.
     *
     * @param source the source
     */
    public void setSource(Source source) {
        this.source = source;
    }
}
