package org.rapidoid.json;

import java.io.OutputStream;
import java.util.Map;

import org.rapidoid.beany.Beany;
import org.rapidoid.log.Log;
import org.rapidoid.util.Cls;
import org.rapidoid.util.U;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

/*
 * #%L
 * rapidoid-json
 * %%
 * Copyright (C) 2014 - 2015 Nikolche Mihajlovski
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

public class JSON {

	public static final ObjectMapper MAPPER = mapper();

	private static ObjectMapper mapper() {
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		return mapper;
	}

	public static String stringify(Object value) {
		try {
			return MAPPER.writeValueAsString(Beany.serialize(value));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public static void stringify(Object value, OutputStream out) {
		try {
			MAPPER.writeValue(out, Beany.serialize(value));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public static <T> T parse(String json, Class<T> valueType) {
		try {
			return MAPPER.readValue(json, valueType);
		} catch (Exception e) {
			Log.error("Cannot parse JSON!", "json", json, "error", e);
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	public static Map<String, Object> parseMap(String json) {
		return parse(json, Map.class);
	}

	public static void warmup() {
		JSON.stringify(123);
		JSON.parse("{}", Map.class);
	}

	public static String save(Object value) {
		try {
			Object ser = Beany.serialize(value);
			Class<?> cls = value != null ? value.getClass() : null;
			Map<String, Object> map = U.map("_", cls.getCanonicalName(), "v", ser);
			return MAPPER.writeValueAsString(map);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	public static Object load(String json) {
		Map<String, Object> map = parseMap(json);
		String clsName = (String) map.get("_");
		Class<Object> type = Cls.getClassIfExists(clsName);
		if (type == null) {
			return null;
		}

		Object ser = map.get("v");

		if (ser instanceof Map) {
			Object value = Cls.newInstance(type);
			Map<String, Object> props = (Map<String, Object>) ser;
			Beany.update(value, props, false);
			return value;
		}

		return ser;
	}
}
