// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io.impls;

import static org.junit.Assert.*;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.Serializable;

import org.apache.log4j.Logger;
import org.junit.Test;
import org.refcodes.component.OpenException;

/**
 * Tests the {@link PrefetchBidirectionalStreamConnectionTransceiverImpl} and the parts
 * {@link PrefetchInputStreamConnectionReceiverImpl} and {@link OutputStreamConnectionSenderImpl}.
 */
public class IoStreamConnectionTest {

	private static Logger LOGGER = Logger.getLogger( IoStreamConnectionTest.class );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String DATAGRAM_A = "Datagram A";
	private static final String DATAGRAM_B = "Datagram B";

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tests the {@link PrefetchBidirectionalStreamConnectionTransceiverImpl}.
	 * 
	 * @throws IOException
	 */
	@Test
	public void testIoStreamTransceiver() throws OpenException, InterruptedException, IOException {

		PipedInputStream theInputStreamA = new PipedInputStream();
		PipedOutputStream theOutputStreamB = new PipedOutputStream( theInputStreamA );
		PipedInputStream theInputStreamB = new PipedInputStream();
		PipedOutputStream theOutputStreamA = new PipedOutputStream( theInputStreamB );

		// OPEN:
		PrefetchBidirectionalStreamConnectionTransceiverImpl<Serializable> theIoStreamTransceiverA = new PrefetchBidirectionalStreamConnectionTransceiverImpl<Serializable>();
		PrefetchBidirectionalStreamConnectionTransceiverImpl<Serializable> theIoStreamTransceiverB = new PrefetchBidirectionalStreamConnectionTransceiverImpl<Serializable>();

		// ---------------------------------------------------------------------
		// Hack to enable single threaded pipe streams:
		// ---------------------------------------------------------------------
		ObjectOutputStream theObjectOutputStreamA = new ObjectOutputStream( theOutputStreamA );
		theObjectOutputStreamA.flush();
		ObjectOutputStream theObjectOutputStreamB = new ObjectOutputStream( theOutputStreamB );
		theObjectOutputStreamB.flush();
		// ---------------------------------------------------------------------

		theIoStreamTransceiverA.open( theInputStreamA, theObjectOutputStreamA );
		theIoStreamTransceiverB.open( theInputStreamB, theObjectOutputStreamB );

		// WRITE FROM A TO B:

		theIoStreamTransceiverA.writeDatagram( DATAGRAM_A );
		String theDatagramA = (String) theIoStreamTransceiverB.readDatagram();
		LOGGER.info( "Transceiver B read datagram \"" + theDatagramA + "\" from transceiver A." );
		assertEquals( DATAGRAM_A, theDatagramA );

		// WRITE FROM B TO A:

		theIoStreamTransceiverB.writeDatagram( DATAGRAM_B );
		String theDatagramB = (String) theIoStreamTransceiverA.readDatagram();
		LOGGER.info( "Transceiver A read datagram \"" + theDatagramB + "\" from transceiver B." );
		assertEquals( DATAGRAM_B, theDatagramB );

		// CLOSE:

		theIoStreamTransceiverA.close();
		theIoStreamTransceiverB.close();

		assertTrue( theIoStreamTransceiverA.isClosed() );
		assertTrue( theIoStreamTransceiverB.isClosed() );

		try {
			theIoStreamTransceiverA.writeDatagram( DATAGRAM_A );
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}

		try {
			theIoStreamTransceiverA.readDatagram();
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}

		try {
			theIoStreamTransceiverB.writeDatagram( DATAGRAM_A );
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}

		try {
			theIoStreamTransceiverB.readDatagram();
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}
	}

	/**
	 * Tests the {@link OutputStreamConnectionSenderImpl} and
	 * {@link PrefetchInputStreamConnectionReceiverImpl}.
	 * 
	 * @throws IOException
	 */
	@Test
	public void testIoStreamSenderAndReceiver() throws OpenException, InterruptedException, IOException {

		PipedInputStream theInputStream = new PipedInputStream();
		PipedOutputStream theOutputStream = new PipedOutputStream( theInputStream );

		// ---------------------------------------------------------------------
		// Hack to enable single threaded pipe streams:
		// ---------------------------------------------------------------------
		ObjectOutputStream theObjectOutputStream = new ObjectOutputStream( theOutputStream );
		theObjectOutputStream.flush();
		// ---------------------------------------------------------------------

		// OPEN:

		PrefetchInputStreamConnectionReceiverImpl<Serializable> theIoStreamReceiver = new PrefetchInputStreamConnectionReceiverImpl<Serializable>();
		theIoStreamReceiver.open( theInputStream );
		OutputStreamConnectionSenderImpl<Serializable> theIoStreamSender = new OutputStreamConnectionSenderImpl<Serializable>();
		theIoStreamSender.open( theObjectOutputStream );

		// WRITE A AND B:

		theIoStreamSender.writeDatagram( DATAGRAM_A );
		theIoStreamSender.writeDatagram( DATAGRAM_B );
		String theDatagram = (String) theIoStreamReceiver.readDatagram();
		LOGGER.info( "Receiver read datagram \"" + theDatagram + "\" from sender." );
		assertEquals( DATAGRAM_A, theDatagram );
		theDatagram = (String) theIoStreamReceiver.readDatagram();
		LOGGER.info( "Receiver read datagram \"" + theDatagram + "\" from sender." );
		assertEquals( DATAGRAM_B, theDatagram );

		// CLOSE:

		theIoStreamSender.close();
		theIoStreamReceiver.close();

		assertTrue( theIoStreamSender.isClosed() );
		assertTrue( theIoStreamReceiver.isClosed() );

		try {
			theIoStreamSender.writeDatagram( DATAGRAM_A );
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}

		try {
			theIoStreamReceiver.readDatagram();
			fail();
		}
		catch ( OpenException aException ) { /* Expected */}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////
}
