// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io.impls;

import static org.junit.Assert.*;

import java.io.Serializable;

import org.apache.log4j.Logger;
import org.junit.Test;
import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * Tests the {@link LoopbackTransceiverImpl} and the parts
 * {@link LoopbackReceiverImpl} and {@link LoopbackSenderImpl}.
 */
public class LoopbackConnectionTest {

	private static Logger LOGGER = Logger.getLogger( LoopbackConnectionTest.class );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String DATAGRAM_A = "Datagram A";
	private static final String DATAGRAM_B = "Datagram B";

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tests the {@link LoopbackTransceiverImpl}.
	 */
	@Test
	public void testLoopbackTransceiver() throws OpenException, InterruptedException, CloseException {

		for ( int i = 0; i < 2; i++ ) {

			// OPEN:

			LoopbackTransceiverImpl<Serializable> theLoopbackTransceiverA = new LoopbackTransceiverImpl<Serializable>();
			LoopbackTransceiverImpl<Serializable> theLoopbackTransceiverB = new LoopbackTransceiverImpl<Serializable>();
			theLoopbackTransceiverA.open( theLoopbackTransceiverB );
			theLoopbackTransceiverB.open( theLoopbackTransceiverA );

			// WRITE FROM A TO B:

			theLoopbackTransceiverA.writeDatagram( DATAGRAM_A );
			String theDatagramA = (String) theLoopbackTransceiverB.readDatagram();
			LOGGER.info( "Transceiver B read datagram \"" + theDatagramA + "\" from transceiver A." );
			assertEquals( DATAGRAM_A, theDatagramA );

			// WRITE FROM B TO A:

			theLoopbackTransceiverB.writeDatagram( DATAGRAM_B );
			String theDatagramB = (String) theLoopbackTransceiverA.readDatagram();
			LOGGER.info( "Transceiver A read datagram \"" + theDatagramB + "\" from transceiver B." );
			assertEquals( DATAGRAM_B, theDatagramB );

			// CLOSE:

			if ( i == 0 ) {
				LOGGER.info( "Closing transceiver A." );
				theLoopbackTransceiverA.close();
				assertTrue( theLoopbackTransceiverA.isClosed() );
			}
			if ( i == 1 ) {
				LOGGER.info( "Closing transceiver B." );
				theLoopbackTransceiverB.close();
				assertTrue( theLoopbackTransceiverB.isClosed() );
			}

			try {
				theLoopbackTransceiverA.writeDatagram( DATAGRAM_A );
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}

			try {
				theLoopbackTransceiverA.readDatagram();
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}

			try {
				theLoopbackTransceiverB.writeDatagram( DATAGRAM_A );
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}

			try {
				theLoopbackTransceiverB.readDatagram();
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}
		}
	}

	/**
	 * Tests the {@link LoopbackSenderImpl} and {@link LoopbackReceiverImpl}.
	 */
	@Test
	public void testLoopbackSenderAndReceiver() throws OpenException, InterruptedException, CloseException {

		for ( int i = 0; i < 2; i++ ) {

			// OPEN:

			LoopbackReceiverImpl<Serializable> theLoopbackReceiver = new LoopbackReceiverImpl<Serializable>();
			LoopbackSenderImpl<Serializable> theLoopbackSender = new LoopbackSenderImpl<Serializable>();
			theLoopbackReceiver.open( theLoopbackSender );
			theLoopbackSender.open( theLoopbackReceiver );

			// WRITE A AND B:

			theLoopbackSender.writeDatagram( DATAGRAM_A );
			theLoopbackSender.writeDatagram( DATAGRAM_B );
			String theDatagram = (String) theLoopbackReceiver.readDatagram();
			LOGGER.info( "Receiver read datagram \"" + theDatagram + "\" from sender." );
			assertEquals( DATAGRAM_A, theDatagram );
			theDatagram = (String) theLoopbackReceiver.readDatagram();
			LOGGER.info( "Receiver read datagram \"" + theDatagram + "\" from sender." );
			assertEquals( DATAGRAM_B, theDatagram );

			// CLOSE:

			if ( i == 0 ) {
				LOGGER.info( "Closing sender." );
				theLoopbackSender.close();
				assertTrue( theLoopbackSender.isClosed() );
			}
			if ( i == 1 ) {
				LOGGER.info( "Closing receiver." );
				theLoopbackReceiver.close();
				assertTrue( theLoopbackReceiver.isClosed() );
			}

			try {
				theLoopbackSender.writeDatagram( DATAGRAM_A );
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}

			try {
				theLoopbackReceiver.readDatagram();
				fail();
			}
			catch ( OpenException aException ) { /* Expected */}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////
}
