// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io.impls;

import static org.junit.Assert.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.zip.ZipException;

import org.apache.log4j.Logger;
import org.junit.Test;
import org.refcodes.data.DelimeterConsts;
import org.refcodes.io.FileUtility;
import org.refcodes.textual.CsvEscapeMode;
import org.refcodes.textual.impls.CsvBuilderImpl;

public class ZipFileStreamTest {

	private static final String ZIP_FILE_MAGIC_BYTES = "PK";
	private static Logger LOGGER = Logger.getLogger( ZipFileStreamTest.class );

	// @formatter:off
	private static final String[][] ZIP_FILE_NAMES = new String[][] {
		{"test.log.zip", "test.log"},
		{"test.log.ZIP", "test.log"},
		{"home/user/test.log.zip", "home/user/test.log"},
		{".zip", ""}
	};
	// @formatter:on

	@Test
	public void toAutoDetectInputStream() {
		for ( int i = 0; i < ZIP_FILE_NAMES.length; i++ ) {
			LOGGER.info( "In:      \"" + ZIP_FILE_NAMES[i][0] + "\"" );
			LOGGER.info( "Out:     \"" + ZipFileInputStreamImpl.toFileNameFromZip( ZIP_FILE_NAMES[i][0] ) + "\"" );
			LOGGER.info( "Expected:\"" + ZIP_FILE_NAMES[i][1] + "\"" );
			assertEquals( ZIP_FILE_NAMES[i][1], ZipFileInputStreamImpl.toFileNameFromZip( ZIP_FILE_NAMES[i][0] ) );
		}
	}

	@Test
	public void toJarHierarchy() throws MalformedURLException {
		URL theJarPath1 = new URL( "jar:file:/home/steiner/Workspaces/com.fightclub/fightclub-app/target/fightclub-app-0.0.1-SNAPSHOT.jar!/webapp/home.xhtml" );
		URL theJarPath2 = new URL( "jar:file:/home/steiner/Workspaces/com.fightclub/fightclub-app/target/fightclub-app-0.0.1-SNAPSHOT.jar!/lib/fightclub-adapter-web-0.0.1-SNAPSHOT.jar!/webapp/home.xhtml" );
		String[] theJarHierarchy;
		String theJarPath;

		theJarHierarchy = FileUtility.toJarHierarchy( theJarPath1 );
		theJarPath = new CsvBuilderImpl().withCsvEscapeMode( CsvEscapeMode.ESCAPED ).withFields( theJarHierarchy ).withDelimiterChar( DelimeterConsts.PATH_DELIMETER ).toRecord();
		LOGGER.info( "JAR parh 1 := " + theJarPath );
		assertEquals( "fightclub-app-0.0.1-SNAPSHOT.jar", theJarPath );

		theJarHierarchy = FileUtility.toJarHierarchy( theJarPath2 );
		theJarPath = new CsvBuilderImpl().withCsvEscapeMode( CsvEscapeMode.ESCAPED ).withFields( theJarHierarchy ).withDelimiterChar( DelimeterConsts.PATH_DELIMETER ).toRecord();
		LOGGER.info( "JAR parh 2 := " + theJarPath );
		assertEquals( "fightclub-app-0.0.1-SNAPSHOT.jar/fightclub-adapter-web-0.0.1-SNAPSHOT.jar", theJarPath );
	}

	@Test
	public void testZipFile() throws ZipException, FileNotFoundException, IOException {
		String theText = "The class <" + getClass().getName() + "> says: \"Hello World!\"";

		// Write ZIP:
		File theZipFile = File.createTempFile( "refcodes-", ".zip" );
		theZipFile.deleteOnExit();
		OutputStream theZipOut = new ZipFileOutputStreamImpl( theZipFile );
		theZipOut.write( theText.getBytes() );
		theZipOut.close();

		// Read ZIP:
		InputStream theZipIn = new ZipFileInputStreamImpl( theZipFile );
		String theZipData = toString( theZipIn );
		theZipIn.close();
		assertEquals( theText, theZipData );

		// Read raw:
		InputStream theRawIn = new FileInputStream( theZipFile );
		String theRawData = toString( theRawIn );
		theRawIn.close();
		assertNotEquals( theZipData, theRawData );
		assertEquals( ZIP_FILE_MAGIC_BYTES, theRawData.substring( 0, 2 ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private String toString( InputStream theZipIn ) throws IOException {
		StringBuffer theBuffer = new StringBuffer();
		int eByte = theZipIn.read();
		while ( eByte != -1 ) {
			theBuffer.append( new String( new byte[] {
					(byte) eByte
			} ) );
			eByte = theZipIn.read();
		}
		return theBuffer.toString();
	}
}
