// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;

import org.refcodes.component.AbstractConnectableAutomaton;
import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionComponent.ConnectionAutomaton;
import org.refcodes.component.ConnectionOpenable;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.exception.ExceptionUtility;

/**
 * Abstract implementation of the {@link ByteReceiver} interface. The
 * {@link #open(InputStream)}, {@link #setConnectionStatus(ConnectionStatus)}
 * and {@link #isOpenable()} methods are your hooks when extending this class.
 */
public abstract class AbstractInputStreamByteReceiver extends AbstractConnectableAutomaton implements ByteReceiver {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private LinkedList<Byte> _datagramQueue = new LinkedList<Byte>();

	private InputStream _inputStream = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////
	@Override
	public boolean hasDatagram() throws OpenException {
		if ( isClosed() ) return false;
		if ( !_datagramQueue.isEmpty() ) return true;
		try {
			int theWord = _inputStream.read();
			if ( theWord == -1 ) {
				close();
				synchronized ( _datagramQueue ) {
					_datagramQueue.notifyAll();
				}
				return false;
			}
			_datagramQueue.add( (byte) (theWord & 0xFF) );
			return true;
		}
		catch ( CloseException e ) {
			synchronized ( _datagramQueue ) {
				_datagramQueue.notifyAll();
			}
			throw new OpenException( "Unable to test datagram availability, connection status is <" + getConnectionStatus() + ">.", e );
		}
		catch ( IOException ioe ) {

			if ( isClosed() ) return false;

			synchronized ( _datagramQueue ) {
				_datagramQueue.notifyAll();
			}
			try {
				if ( ExceptionUtility.isThrownAsOfAlreadyClosed( ioe ) ) {
					super.close();
				}
				else {
					close();
				}
			}
			catch ( CloseException e ) {
				throw new OpenException( "Unable to test datagram availability, connection status is <" + getConnectionStatus() + ">.", e );
			}
			throw new OpenException( "Unable to test datagram availability, connection status is <" + getConnectionStatus() + ">.", ioe );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public byte readDatagram() throws OpenException, InterruptedException {
		if ( _datagramQueue.isEmpty() && !isOpened() ) {
			throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is <" + getConnectionStatus() + ">." );
		}
		if ( !hasDatagram() ) {
			throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is <" + getConnectionStatus() + ">." );
		}
		Byte theDatagram = _datagramQueue.poll();
		while ( theDatagram == null ) {
			if ( !hasDatagram() ) {
				throw new OpenException( "Unable to read datagram  as the connection is NOT OPEN; connection status is <" + getConnectionStatus() + ">." );
			}
		}
		return theDatagram;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void releaseAll() {
		synchronized ( _datagramQueue ) {
			_datagramQueue.notifyAll();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close() throws CloseException {
		if ( !isClosed() ) {
			super.close();
			try {
				_inputStream.close();
			}
			catch ( IOException e ) {
				if ( !ExceptionUtility.isThrownAsOfAlreadyClosed( e ) ) throw new CloseException( "Unable to close receiver, connection status is <" + getConnectionStatus() + ">.", e );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Open, see {@link ConnectionOpenable#open(Object)}.
	 *
	 * @param aInputStream the input stream
	 * @throws OpenException the open exception
	 */
	protected synchronized void open( InputStream aInputStream ) throws OpenException {
		if ( isOpened() ) {
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is <" + getConnectionStatus() + ">." );
		}
		_inputStream = aInputStream;
		// LOGGER.debug( "Starting I/O stream receiver daemon <" +
		// _ioStreamReceiverDaemon.getClass().getName() + ">." );
		setConnectionStatus( ConnectionStatus.OPENED );
	}

	/**
	 * Checks if is openable. See also
	 * {@link ConnectionAutomaton#isOpenable(Object)}.
	 *
	 * @param aInputStream the input stream
	 * @return true, if is openable
	 */
	protected boolean isOpenable( InputStream aInputStream ) {
		if ( aInputStream == null ) {
			return false;
		}
		return !isOpened();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
