// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;

import org.refcodes.component.OpenException;

/**
 * The {@link BlockConsumer} is used to send datagram blocks (arrays) in a
 * unified way.
 *
 * @param <DATA> The type of the datagram block (array) to be operated with. Do
 *        not provide an array type as the methods use to generic type for
 *        defining an array argument.
 */
@FunctionalInterface
public interface BlockConsumer<DATA extends Serializable> {

	/**
	 * Writes (sends) a datagram block to a listening {@link DatagramReceiver}
	 * or {@link BlockReceiver}.
	 * 
	 * @param aDatagram The datagram to be pushed to the receiving
	 *        {@link DatagramReceiver} or {@link BlockReceiver}.
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	default void writeDatagrams( DATA[] aDatagram ) throws OpenException {
		writeDatagrams( aDatagram, 0, aDatagram.length );
	}

	/**
	 * Write datagrams.
	 *
	 * @param aDatagram the datagram
	 * @param aOffset the offset
	 * @param aLength the length
	 * @throws OpenException the open exception
	 */
	void writeDatagrams( DATA[] aDatagram, int aOffset, int aLength ) throws OpenException;

}
