// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;

/**
 * Implementation of the {@link OutputStreamConnectionSender} interface.
 *
 * @param <DATA> The type of the datagram to be operated with.
 */
public class OutputStreamConnectionSenderImpl<DATA extends Serializable> extends AbstractSender<DATA> implements OutputStreamConnectionSender<DATA> {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private ObjectOutputStream _objectOutputStream = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void open( OutputStream aOutputStream ) throws OpenException {
		if ( isOpened() ) {
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is " + getConnectionStatus() + "." );
		}
		try {
			if ( aOutputStream instanceof ObjectOutputStream ) {
				// -------------------------------------------------------------
				// Hack to enable single threaded pipe streams:
				// -------------------------------------------------------------
				_objectOutputStream = (ObjectOutputStream) aOutputStream;
			}
			else {
				if ( !(aOutputStream instanceof BufferedOutputStream) ) {
					_objectOutputStream = new ObjectOutputStream( new BufferedOutputStream( aOutputStream ) );
				}
				else {
					_objectOutputStream = new ObjectOutputStream( aOutputStream );
				}
			}
			_objectOutputStream.flush();
		}
		catch ( IOException aException ) {
			throw new OpenException( "Unable to open the I/O stream receiver as of a causing exception.", aException );
		}
		setConnectionStatus( ConnectionStatus.OPENED );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpenable( OutputStream aOutputStream ) {
		if ( aOutputStream == null ) {
			return false;
		}
		return !isOpened();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		try {
			_objectOutputStream.flush();
		}
		catch ( IOException e ) {
			throw new OpenException( "Unable to flush underlying output stream <" + _objectOutputStream + ">.", e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close() throws CloseException {
		super.close();
		try {
			_objectOutputStream.close();
		}
		catch ( IOException aException ) {}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagram( DATA aDatagram ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagram.getClass().getName() + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		synchronized ( _objectOutputStream ) {
			try {
				_objectOutputStream.writeObject( aDatagram );
				_objectOutputStream.flush();
			}
			catch ( IOException aException ) {
				if ( isClosed() ) {
					return;
				}
				else {
					try {
						close();
					}
					catch ( CloseException e ) {
						throw new OpenException( "Unable to close malfunctioning connection.", e );
					}
				}
				throw new OpenException( aException );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

}
