// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.zip.ZipException;

import org.refcodes.tabular.CsvRecordsReader;
import org.refcodes.tabular.Header;

/**
 * Represents an {@link InputStream} from a provided {@link File}: In case the
 * file points to a ZIP compressed file, then the uncompressed data of the
 * therein contained file with the same name excluding the ".zip" extension is
 * provided by the {@link InputStream}.
 */
public class ZipFileInputStream extends BufferedInputStream {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new zip file input stream impl.
	 *
	 * @param parent the parent
	 * @param child the child
	 * @throws ZipException the zip exception
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public ZipFileInputStream( File parent, String child ) throws ZipException, FileNotFoundException, IOException {
		this( new File( parent, child ) );
	}

	/**
	 * Instantiates a new zip file input stream impl.
	 *
	 * @param parent the parent
	 * @param child the child
	 * @throws ZipException the zip exception
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public ZipFileInputStream( String parent, String child ) throws ZipException, FileNotFoundException, IOException {
		this( new File( parent, child ) );
	}

	/**
	 * Instantiates a new zip file input stream impl.
	 *
	 * @param pathname the pathname
	 * @throws ZipException the zip exception
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public ZipFileInputStream( String pathname ) throws ZipException, FileNotFoundException, IOException {
		this( new File( pathname ) );
	}

	/**
	 * Instantiates a new zip file input stream impl.
	 *
	 * @param uri the uri
	 * @throws ZipException the zip exception
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public ZipFileInputStream( URI uri ) throws ZipException, FileNotFoundException, IOException {
		this( new File( uri ) );
	}

	/**
	 * Instantiates a new zip file input stream impl.
	 *
	 * @param aFile the file
	 * @throws ZipException the zip exception
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public ZipFileInputStream( File aFile ) throws ZipException, FileNotFoundException, IOException {
		super( toInputStream( aFile ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns an {@link InputStream} from the provided {@link File}. In case
	 * the file points to a ZIP compressed file, then the uncompressed data is
	 * provided by the {@link InputStream}.
	 * 
	 * @param aFile The {@link File} for which to get the {@link InputStream}.
	 * 
	 * @return An {@link InputStream}, in case of a ZIP compressed {@link File},
	 *         an uncompressed {@link InputStream} is returned.
	 * 
	 * @throws ZipException in case there were problems when accessing the ZIP
	 *         compressed {@link File}.
	 * 
	 * @throws IOException in case there were problems working with the
	 *         {@link File}.
	 * 
	 * @throws FileNotFoundException in case there was none such {@link File}
	 *         found.
	 */
	protected static InputStream toInputStream( File aFile ) throws ZipException, IOException, FileNotFoundException {
		return ZipFileUtility.toInputStream( aFile );
	}

	/**
	 * Truncates the ".zip" suffix from the filename and returns the result. For
	 * example a file with name "log-2023-07-12.txt.zip" results in
	 * "log-2023-07-12.txt".
	 * 
	 * @param aZipFileName The file name of the ZIP file for which to get the
	 *        "inner" file name.
	 * 
	 * @return The "inner" file name if the file suffix was ".zip", else null.
	 */
	protected static String toFileNameFromZip( String aZipFileName ) {
		return ZipFileUtility.toFileNameFromZip( aZipFileName );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To reduce code duplication I move the according methods to the depending
	 * class and make them accessible via the protected modifier. This depending
	 * class is extended by this {@link ZipFileUtility} class which enables
	 * access to the {@link ZipFileInputStream} class.
	 */
	@SuppressWarnings("rawtypes")
	private static class ZipFileUtility extends CsvRecordsReader {

		/**
		 * Instantiates a new zip file utility.
		 *
		 * @param aHeader the header
		 * @param aCsvFile the csv file
		 * @throws FileNotFoundException the file not found exception
		 * @throws IOException Signals that an I/O exception has occurred.
		 * @throws ZipException the zip exception
		 */
		@SuppressWarnings("unchecked")
		private ZipFileUtility( Header aHeader, File aCsvFile ) throws FileNotFoundException, IOException, ZipException {
			super( aHeader, aCsvFile );
		}

		/**
		 * To input stream.
		 *
		 * @param aFile the file
		 * @return the input stream
		 * @throws ZipException the zip exception
		 * @throws IOException Signals that an I/O exception has occurred.
		 * @throws FileNotFoundException the file not found exception
		 */
		protected static InputStream toInputStream( File aFile ) throws ZipException, IOException, FileNotFoundException {
			return CsvRecordsReader.toInputStream( aFile );
		}

		/**
		 * To file name from zip.
		 *
		 * @param aZipFileName the zip file name
		 * @return the string
		 */
		protected static String toFileNameFromZip( String aZipFileName ) {
			return CsvRecordsReader.toFileNameFromZip( aZipFileName );
		}
	}
}
