// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.logger.alt.slf4j.impls;

import org.refcodes.logger.Logger;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.consts.LogPriority;
import org.refcodes.logger.consts.LoggerFields;
import org.refcodes.logger.impls.RuntimeLoggerImpl;
import org.refcodes.logger.impls.RuntimeLoggerSingleton;
import org.refcodes.logger.traps.IllegalRecordRuntimeException;
import org.refcodes.logger.traps.UnexpectedLogRuntimeException;
import org.refcodes.runtime.utils.RuntimeUtility;
import org.refcodes.tabular.Record;
import org.refcodes.tabular.traps.ColumnMismatchException;
import org.slf4j.LoggerFactory;

/**
 * @author steiner
 *
 */
public class Slf4jLoggerImpl implements Logger<Object> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private transient org.slf4j.Logger _logger;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	public Slf4jLoggerImpl() {
		Class<?> theCallees[] = new Class<?>[] {
				RuntimeLoggerImpl.class, RuntimeLogger.class, RuntimeLoggerSingleton.class, Slf4jLoggerImpl.class
		};
		Class<? extends StackTraceElement> theCaller = RuntimeUtility.getCallerStackTraceElement( theCallees ).getClass();
		_logger = LoggerFactory.getLogger( theCaller );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void log( Record<? extends Object> aRecord ) throws IllegalRecordRuntimeException, UnexpectedLogRuntimeException {
		try {
			LogPriority thePriority = (LogPriority) LoggerFields.LOG_PRIORITY.getColumn().get( aRecord );
			Integer theLineNumber = (Integer) LoggerFields.LOG_LINE_NUMBER.getColumn().get( aRecord );

			String theMessage = (String) LoggerFields.LOG_MESSAGE.getColumn().get( aRecord );
			String theMethodName = (String) LoggerFields.LOG_METHODE_NAME.getColumn().get( aRecord );
			Exception theException = (Exception) LoggerFields.LOG_EXCEPTION.getColumn().get( aRecord );
			String theLogMessage = theLineNumber + " " + theMessage + "@" + theMethodName;

			switch ( thePriority ) {
			case PANIC:
				_logger.error( theLogMessage, theException );
				break;
			case ALERT:
				_logger.error( theLogMessage, theException );
				break;
			case CRITICAL:
				_logger.error( theLogMessage, theException );
				break;
			case ERROR:
				_logger.error( theLogMessage, theException );
				break;
			case WARN:
				_logger.warn( theLogMessage, theException );
				break;
			case NOTICE:
				_logger.info( theLogMessage, theException );
				break;
			case INFO:
				_logger.info( theLogMessage, theException );
				break;
			case DEBUG:
				_logger.debug( theLogMessage, theException );
				break;
			case TRACE:
				_logger.trace( theLogMessage, theException );
				break;
			case NONE:
				break;
			case DISCARD:
				break;
			}
		}
		catch ( ColumnMismatchException | ClassCastException e ) {
			throw new IllegalRecordRuntimeException( aRecord, e );
		}
	}
}