// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.logger.alt.slf4j.impls;

import java.text.MessageFormat;

import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.consts.LogPriority;
import org.refcodes.logger.mixins.RuntimeLoggerAccessor;
import org.refcodes.runtime.utils.RuntimeUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author steiner
 *
 */
public class Slf4jRuntimeLoggerImpl implements RuntimeLogger {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private transient Logger _logger = null;
	private transient RuntimeLogger _runtimeLogger = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * In case the {@link Slf4jRuntimeLoggerImpl} (as created by the
	 * {@link Slf4jRuntimeLoggerFactorySingleton}) detects that SLF4J has bound
	 * a {@link Slf4jRuntimeLoggerAdapterImpl} (i.e. the REFCODES.ORG SLF4J
	 * binding), it directly delegates its method calls to the wrapped
	 * {@link RuntimeLogger} instead of marshaling a log request through the
	 * {@link Slf4jRuntimeLoggerAdapterImpl}; as marshaling would mean
	 * consolidating of various detailed {@link LogPriority} levels to a single
	 * SLF4J log level.
	 */
	public Slf4jRuntimeLoggerImpl() {
		String className = RuntimeUtility.getCallerStackTraceElement( Slf4jRuntimeLoggerFactoryImpl.class ).getClassName();
		Logger aLogger = LoggerFactory.getLogger( className );
		toRuntimeLogger( aLogger );
	}

	/**
	 * In case the {@link Slf4jRuntimeLoggerImpl} (as created by the
	 * {@link Slf4jRuntimeLoggerFactorySingleton}) detects that SLF4J has bound
	 * a {@link Slf4jRuntimeLoggerAdapterImpl} (i.e. the REFCODES.ORG SLF4J
	 * binding), it directly delegates its method calls to the wrapped
	 * {@link RuntimeLogger} instead of marshaling a log request through the
	 * {@link Slf4jRuntimeLoggerAdapterImpl}; as marshaling would mean
	 * consolidating of various detailed {@link LogPriority} levels to a single
	 * SLF4J log level.
	 * 
	 * @param A SLF4J logger which might be of type
	 *        {@link Slf4jRuntimeLoggerAdapterImpl} in which case the therein
	 *        wrapped {@link RuntimeLogger} would be addressed directly using
	 *        the {@link Slf4jRuntimeLoggerAdapterImpl#getRuntimeLogger()}
	 *        method.
	 */
	public Slf4jRuntimeLoggerImpl( Logger aLogger ) {
		toRuntimeLogger( aLogger );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void log( LogPriority aPriority, String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.log( aPriority, aMessage );
		else {
			switch ( aPriority ) {
			case PANIC:
				_logger.error( aMessage );
				break;
			case ALERT:
				_logger.error( aMessage );
				break;
			case CRITICAL:
				_logger.error( aMessage );
				break;
			case ERROR:
				_logger.error( aMessage );
				break;
			case WARN:
				_logger.warn( aMessage );
				break;
			case NOTICE:
				_logger.info( aMessage );
				break;
			case INFO:
				_logger.info( aMessage );
				break;
			case DEBUG:
				_logger.debug( aMessage );
				break;
			case TRACE:
				_logger.trace( aMessage );
				break;
			case DISCARD:
				break;
			case NONE:
				_logger.info( aMessage );
				break;
			default:
				_logger.info( aMessage );
				break;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void log( LogPriority aPriority, String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.log( aPriority, aMessage, aArguments );
		else {
			switch ( aPriority ) {
			case PANIC:
				_logger.error( aMessage, aArguments );
				break;
			case ALERT:
				_logger.error( aMessage, aArguments );
				break;
			case CRITICAL:
				_logger.error( aMessage, aArguments );
				break;
			case ERROR:
				_logger.error( aMessage, aArguments );
				break;
			case WARN:
				_logger.warn( aMessage, aArguments );
				break;
			case NOTICE:
				_logger.info( aMessage, aArguments );
				break;
			case INFO:
				_logger.info( aMessage, aArguments );
				break;
			case DEBUG:
				_logger.debug( aMessage, aArguments );
				break;
			case TRACE:
				_logger.trace( aMessage, aArguments );
				break;
			case DISCARD:
				break;
			case NONE:
				_logger.info( aMessage, aArguments );
				break;
			default:
				_logger.info( aMessage, aArguments );
				break;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		if ( _runtimeLogger != null ) return _runtimeLogger.getName();
		else {
			return _logger.getName();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public LogPriority getLogPriority() {
		if ( _runtimeLogger != null ) return _runtimeLogger.getLogPriority();
		else {
			if ( _logger.isDebugEnabled() ) return LogPriority.DEBUG;
			if ( _logger.isErrorEnabled() ) return LogPriority.ERROR;
			if ( _logger.isInfoEnabled() ) return LogPriority.INFO;
			if ( _logger.isTraceEnabled() ) return LogPriority.TRACE;
			if ( _logger.isWarnEnabled() ) return LogPriority.WARN;
			return LogPriority.NONE;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void log( LogPriority aPriority, String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.log( aPriority, aMessage, aThrowable );
		else {
			switch ( aPriority ) {
			case PANIC:
				_logger.error( aMessage, aThrowable );
				break;
			case ALERT:
				_logger.error( aMessage, aThrowable );
				break;
			case CRITICAL:
				_logger.error( aMessage, aThrowable );
				break;
			case ERROR:
				_logger.error( aMessage, aThrowable );
				break;
			case WARN:
				_logger.warn( aMessage, aThrowable );
				break;
			case NOTICE:
				_logger.info( aMessage, aThrowable );
				break;
			case INFO:
				_logger.info( aMessage, aThrowable );
				break;
			case DEBUG:
				_logger.debug( aMessage, aThrowable );
				break;
			case TRACE:
				_logger.trace( aMessage, aThrowable );
				break;
			case DISCARD:
				break;
			case NONE:
				_logger.info( aMessage, aThrowable );
				break;
			default:
				_logger.info( aMessage, aThrowable );
				break;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void log( LogPriority aPriority, String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.log( aPriority, aMessage, aThrowable, aArguments );
		else {
			switch ( aPriority ) {
			case PANIC:
				if ( _logger.isErrorEnabled() ) _logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case ALERT:
				if ( _logger.isErrorEnabled() ) _logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case CRITICAL:
				if ( _logger.isErrorEnabled() ) _logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case ERROR:
				if ( _logger.isErrorEnabled() ) _logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case WARN:
				if ( _logger.isWarnEnabled() ) _logger.warn( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case NOTICE:
				if ( _logger.isInfoEnabled() ) _logger.info( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case INFO:
				if ( _logger.isInfoEnabled() ) _logger.info( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case DEBUG:
				if ( _logger.isDebugEnabled() ) _logger.debug( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case TRACE:
				if ( _logger.isTraceEnabled() ) _logger.trace( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			case DISCARD:
				break;
			case NONE:
				if ( _logger.isInfoEnabled() ) _logger.info( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			default:
				if ( _logger.isInfoEnabled() ) _logger.info( MessageFormat.format( aMessage, aArguments ), aThrowable );
				break;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLog( LogPriority aPriority ) {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLog( aPriority );
		else {
			return aPriority.getPriority() >= getLogPriority().getPriority() && aPriority.getPriority() >= 0;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void trace( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.trace( aMessage );
		else {
			_logger.trace( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void trace( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.trace( aMessage, aArguments );
		else {
			_logger.trace( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogTrace() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogTrace();
		else {
			return isLog( LogPriority.TRACE );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void debug( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.debug( aMessage );
		else {
			_logger.debug( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void debug( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.debug( aMessage, aArguments );
		else {
			_logger.debug( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogDebug() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogDebug();
		else {
			return isLog( LogPriority.DEBUG );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void info( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.info( aMessage );
		else {
			_logger.info( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void info( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.info( aMessage, aArguments );
		else {
			_logger.info( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogInfo() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogInfo();
		else {
			return isLog( LogPriority.INFO );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void notice( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.notice( aMessage );
		else {
			_logger.info( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void notice( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.notice( aMessage, aArguments );
		else {
			_logger.info( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogNotice() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogNotice();
		else {
			return isLog( LogPriority.NOTICE );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void warn( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.warn( aMessage );
		else {
			_logger.warn( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void warn( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.warn( aMessage, aArguments );
		else {
			_logger.warn( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void warn( String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.warn( aMessage, aThrowable );
		else {
			_logger.warn( aMessage, aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void warn( String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.warn( aMessage, aArguments );
		else {
			_logger.warn( MessageFormat.format( aMessage, aArguments ), aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogWarn() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogWarn();
		else {
			return isLog( LogPriority.WARN );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void error( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.error( aMessage );
		else {
			_logger.error( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void error( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.error( aMessage, aArguments );
		else {
			_logger.error( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void error( String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.error( aMessage, aThrowable );
		else {
			_logger.error( aMessage, aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void error( String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.error( aMessage, aThrowable, aArguments );
		else {
			_logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogError() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogError();
		else {
			return isLog( LogPriority.ERROR );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void critical( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.critical( aMessage );
		else {
			_logger.error( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void critical( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.critical( aMessage, aArguments );
		else {
			_logger.error( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void critical( String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.critical( aMessage, aThrowable );
		else {
			_logger.error( aMessage, aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void critical( String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.critical( aMessage, aThrowable, aArguments );
		else {
			_logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogCritical() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogCritical();
		else {
			return isLog( LogPriority.CRITICAL );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void alert( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.alert( aMessage );
		else {
			_logger.error( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void alert( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.alert( aMessage, aArguments );
		else {
			_logger.error( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void alert( String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.alert( aMessage, aThrowable );
		else {
			_logger.error( aMessage, aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void alert( String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.alert( aMessage, aThrowable, aArguments );
		else {
			_logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogAlert() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogAlert();
		else {
			return isLog( LogPriority.ALERT );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void panic( String aMessage ) {
		if ( _runtimeLogger != null ) _runtimeLogger.panic( aMessage );
		else {
			_logger.error( aMessage );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void panic( String aMessage, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.panic( aMessage, aArguments );
		else {
			_logger.error( aMessage, aArguments );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void panic( String aMessage, Throwable aThrowable ) {
		if ( _runtimeLogger != null ) _runtimeLogger.panic( aMessage, aThrowable );
		else {
			_logger.error( aMessage, aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void panic( String aMessage, Throwable aThrowable, Object... aArguments ) {
		if ( _runtimeLogger != null ) _runtimeLogger.panic( aMessage, aThrowable, aArguments );
		else {
			_logger.error( MessageFormat.format( aMessage, aArguments ), aThrowable );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLogPanic() {
		if ( _runtimeLogger != null ) return _runtimeLogger.isLogPanic();
		else {
			return isLog( LogPriority.PANIC );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tries to extract an encapsulated {@link RuntimeLogger} instance.
	 */
	private void toRuntimeLogger( Logger aLogger ) {
		if ( (aLogger instanceof RuntimeLoggerAccessor) && ((RuntimeLoggerAccessor) aLogger).getRuntimeLogger() != null ) {
			// -----------------------------------------------------------------
			// To avoid loss of priority level detail, use underlying
			// RuntimeLogger where possible, the Slf4jRuntimeLoggerAdapterImpl
			// provides access to the wrapped RuntimeLogger:
			// -----------------------------------------------------------------
			_runtimeLogger = ((RuntimeLoggerAccessor) aLogger).getRuntimeLogger();
		}
		else {
			_logger = aLogger;
		}
	}
}
