// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

import org.refcodes.generator.UniqueIdGeneratorImpl;
import org.refcodes.generator.UniqueIdGeneratorSingleton;

/**
 * Straight forward implementation of the minimal {@link EventMetaData}
 * interface. You might work in your domain driven framework with according
 * sub-types of the {@link EventMetaData} interface and the according
 * implementations of those.
 */
public class EventMetaDataImpl implements EventMetaData {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected String _uid;
	protected String _alias;
	protected String _group;
	protected String _channel;
	protected Class<?> _publisherType;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link EventMetaData} with no properties set except the
	 * Universal-ID. The Universal-ID is set by invoking the
	 * {@link UniqueIdGeneratorSingleton}'s {@link UniqueIdGeneratorImpl#next()}
	 * method.
	 */
	public EventMetaDataImpl() {
		_uid = UniqueIdGeneratorSingleton.getInstance().next();
	}

	/**
	 * Constructs the {@link EventMetaData} with the given properties. The
	 * Universal-ID is set by invoking the {@link UniqueIdGeneratorSingleton}'s
	 * {@link UniqueIdGeneratorImpl#next()} method.
	 * 
	 * @param aPublisherType The type of the publisher.
	 */
	public EventMetaDataImpl( Class<?> aPublisherType ) {
		_uid = UniqueIdGeneratorSingleton.getInstance().next();
		_publisherType = aPublisherType;
	}

	/**
	 * Constructs the {@link EventMetaData} with the given properties. The
	 * Universal-ID is set by invoking the {@link UniqueIdGeneratorSingleton}'s
	 * {@link UniqueIdGeneratorImpl#next()} method.
	 * 
	 * @param aAlias The alias for the {@link EventMetaData}.
	 * @param aGroup The group for the {@link EventMetaData}.
	 * @param aChannel The channel for the {@link EventMetaData}.
	 */
	public EventMetaDataImpl( String aAlias, String aGroup, String aChannel ) {
		_alias = aAlias;
		_group = aGroup;
		_channel = aChannel;
		_uid = UniqueIdGeneratorSingleton.getInstance().next();
	}

	/**
	 * Constructs the {@link EventMetaData} with the given properties. The
	 * Universal-ID is set by invoking the {@link UniqueIdGeneratorSingleton}'s
	 * {@link UniqueIdGeneratorImpl#next()} method.
	 * 
	 * @param aAlias The alias for the {@link EventMetaData}.
	 * @param aGroup The group for the {@link EventMetaData}.
	 */
	public EventMetaDataImpl( String aAlias, String aGroup ) {
		_alias = aAlias;
		_group = aGroup;
		_uid = UniqueIdGeneratorSingleton.getInstance().next();
	}

	/**
	 * Constructs the {@link EventMetaData} with the given properties. The
	 * Universal-ID is set by invoking the {@link UniqueIdGeneratorSingleton}'s
	 * {@link UniqueIdGeneratorImpl#next()} method.
	 * 
	 * @param aChannel The channel for the {@link EventMetaData}.
	 */
	public EventMetaDataImpl( String aChannel ) {
		_channel = aChannel;
		_uid = UniqueIdGeneratorSingleton.getInstance().next();
	}

	/**
	 * Constructs the {@link EventMetaData} with the given properties.
	 * 
	 * @param aAlias The alias for the {@link EventMetaData}.
	 * @param aGroup The group for the {@link EventMetaData}.
	 * @param aChannel The channel for the {@link EventMetaData}.
	 * @param aUid The Universal-ID for the {@link EventMetaData}.
	 * @param aPublisherType The name for the {@link EventMetaData}.
	 */
	public EventMetaDataImpl( String aAlias, String aGroup, String aChannel, String aUid, Class<?> aPublisherType ) {
		_alias = aAlias;
		_group = aGroup;
		_channel = aChannel;
		_uid = aUid;
		_publisherType = aPublisherType;
	}

	/**
	 * Constructs the {@link EventMetaData} with the properties provided by the
	 * given {@link EventMetaData} instance.
	 * 
	 * @param aMetaData The {@link EventMetaData} instance from which to get the
	 *        properties.
	 */
	public EventMetaDataImpl( EventMetaData aMetaData ) {
		_alias = aMetaData.getAlias();
		_group = aMetaData.getGroup();
		_channel = aMetaData.getChannel();
		_uid = aMetaData.getUniversalId();
		_publisherType = aMetaData.getPublisherType();
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getUniversalId() {
		return _uid;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getAlias() {
		return _alias;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getGroup() {
		return _group;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getChannel() {
		return _channel;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<?> getPublisherType() {
		return _publisherType;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "EventMetaData [Universal-ID=" + _uid + ", name=" + _alias + ", group=" + _group + ", channel=" + _channel + ", publisherType=" + _publisherType + "]";
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

}
