// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatusAccessor;
import org.refcodes.component.LinkComponent.LinkComponentBuilder;
import org.refcodes.component.OpenException;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpMethodAccessor.HttpMethodBuilder;
import org.refcodes.net.HttpMethodAccessor.HttpMethodProperty;

/**
 * An {@link RestEndpointBuilder} extends an {@link RestEndpoint} with builder
 * functionality and adds <code>lambda</code> support for handling the requests
 * addressed to this {@link RestEndpoint}. The <code>lambda</code> defined as
 * {@link RestRequestObserver} acts as the single listener to this
 * {@link RestEndpoint} responsible for handling the requests for which this
 * {@link RestEndpoint} is responsible.
 * 
 * The locator for which an {@link RestEndpointBuilder} is responsible for is
 * defined by the {@link RestEndpointBuilder}'s Locator-Pattern:
 * 
 * A single asterisk ("*") matches zero or more characters within a locator
 * name. A double asterisk ("**") matches zero or more characters across
 * directory levels. A question mark ("?") matches exactly one character within
 * a locator name.
 * 
 * The single asterisk ("*"), the double asterisk ("**") and the question mark
 * ("?") we refer to as wildcard: You get an array with the substitutes of the
 * wildcards using the method {@link RestRequestEvent#getWildcardReplacements()}
 * .
 * 
 * You may name a wildcard by prefixing it with "{someWildcardName}". For
 * example a named wildcard may look as follows: "{arg1}*" or "{arg2}**" or
 * "{arg3}?". You can get the text substituting a named wildcard using the
 * method {@link RestRequestEvent#getWildcardReplacement(String)}.
 */
public interface RestEndpointBuilder extends RestEndpoint, HttpMethodProperty, HttpMethodBuilder<RestEndpointBuilder>, LinkComponentBuilder<RestEndpointBuilder>, ConnectionStatusAccessor {

	/**
	 * With http method.
	 *
	 * @param aHttpMethod the a http method
	 * @return the rest endpoint builder
	 */
	@Override
	default RestEndpointBuilder withHttpMethod( HttpMethod aHttpMethod ) {
		setHttpMethod( aHttpMethod );
		return this;
	}

	/**
	 * Sets the resource locator as of the Builder-Pattern. The locator may
	 * contain wildcards as known from file-systems as follows:
	 * 
	 * A single asterisk ("*") matches zero or more characters within a locator
	 * name. A double asterisk ("**") matches zero or more characters across
	 * directory levels. A question mark ("?") matches exactly one character
	 * within a locator name.
	 * 
	 * You may name a wildcard by prefixing it with "${someWildcardName}=". For
	 * example a named wildcard may look as follows: "${arg1}=*" or "${arg2}=**"
	 * or "${arg3}=?".
	 *
	 * @param aLocatorPattern the a locator pattern
	 * @return The {@link RestEndpoint} builder to continue configuration (as of
	 *         the Builder-Pattern).
	 */
	default RestEndpointBuilder withLocatorPattern( String aLocatorPattern ) {
		setLocatorPattern( aLocatorPattern );
		return this;
	}

	/**
	 * Sets the resource locator. The locator may contain wildcards as known
	 * from file-systems as follows:
	 * 
	 * A single asterisk ("*") matches zero or more characters within a locator
	 * name. A double asterisk ("**") matches zero or more characters across
	 * directory levels. A question mark ("?") matches exactly one character
	 * within a locator name.
	 * 
	 * You may name a wildcard by prefixing it with "${someWildcardName}=". For
	 * example a named wildcard may look as follows: "${arg1}=*" or "${arg2}=**"
	 * or "${arg3}=?".
	 *
	 * @param aLocatorPattern the new locator pattern
	 */
	void setLocatorPattern( String aLocatorPattern );

	/**
	 * Retrieves the {@link RestRequestObserver} to which any requests are
	 * delegated upon invocation of the
	 * {@link #onRequest(RestRequestEvent, org.refcodes.net.HttpServerResponse)}
	 * method.
	 * 
	 * @return The (user defined) {@link RestRequestObserver} to handle
	 *         requests.
	 */
	RestRequestObserver getRequestObserver();

	/**
	 * Sets the {@link RestRequestObserver} to which any requests are delegated
	 * upon invocation of the
	 * {@link #onRequest(RestRequestEvent, org.refcodes.net.HttpServerResponse)}
	 * method.
	 * 
	 * @param aLambda The (user defined) {@link RestRequestObserver} to handle
	 *        requests, feel free to code it as <code>lambda</code> expression!
	 */
	void setRequestObserver( RestRequestObserver aLambda );

	/**
	 * Builder method for setting the {@link RestRequestObserver}.
	 * 
	 * @param aLambda The (user defined) {@link RestRequestObserver} to handle
	 *        requests, feel free to code it as <code>lambda</code> expression
	 * 
	 * @return The {@link RestEndpointBuilder} for the sake of a fluent API.
	 */
	default RestEndpointBuilder withRequestObserver( RestRequestObserver aLambda ) {
		setRequestObserver( aLambda );
		return this;
	}

	/**
	 * With open.
	 *
	 * @return the rest endpoint builder
	 * @throws OpenException the open exception
	 */
	@Override
	default RestEndpointBuilder withOpen() throws OpenException {
		open();
		return this;
	}

	/**
	 * With close.
	 *
	 * @return the rest endpoint builder
	 * @throws CloseException the close exception
	 */
	@Override
	default RestEndpointBuilder withClose() throws CloseException {
		close();
		return this;
	}

	/**
	 * With close quietly.
	 *
	 * @return the rest endpoint builder
	 */
	@Override
	default RestEndpointBuilder withCloseQuietly() {
		closeQuietly();
		return this;
	}

	/**
	 * With close in.
	 *
	 * @param aCloseInMillis the a close in millis
	 * @return the rest endpoint builder
	 */
	@Override
	default RestEndpointBuilder withCloseIn( int aCloseInMillis ) {
		closeIn( aCloseInMillis );
		return this;
	}

}
