// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.rest.impls;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.refcodes.component.ConnectionStatus;
import org.refcodes.controlflow.ControlFlowUtility;
import org.refcodes.data.DaemonLoopSleepTime;
import org.refcodes.data.Text;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.ResponseHeaderFields;
import org.refcodes.net.impls.ApplicationFormFactory;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.ApplicationXmlFactory;
import org.refcodes.net.impls.HttpClientRequestImpl;
import org.refcodes.net.impls.TextPlainFactory;
import org.refcodes.rest.RestCallerBuilder;
import org.refcodes.rest.RestClient;
import org.refcodes.rest.RestRequestBuilder;
import org.refcodes.rest.RestRequestHandler;
import org.refcodes.rest.RestResponse;
import org.refcodes.rest.RestResponseObserver;
import org.refcodes.runtime.Correlation;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;

/**
 * Abstract base implementation of the {@link RestClient} interface being the
 * foundation for various {@link RestClient} implementations such as
 * {@link HttpRestClientImpl} or {@link LoopbackRestClientImpl}.
 * 
 * The {@link AbstractRestClient} is pre-configured with the following
 * {@link MediaTypeFactory} instances:
 * 
 * <ul>
 * <li>{@link ApplicationJsonFactory}</li>
 * <li>{@link ApplicationXmlFactory}</li>
 * <li>{@link TextPlainFactory}</li>
 * <li>{@link ApplicationFormFactory}</li>
 * </ul>
 * 
 * In your sub-classes, overwrite the method {@link #initMedaTypeFactories()},
 * therein calling {@link #addMediaTypeFactory(MediaTypeFactory)} to add (by
 * also invoking super's {@link #initMedaTypeFactories()}) or to set your own
 * (without invoking super's {@link #initMedaTypeFactories()})
 * {@link MediaTypeFactory} instances.
 */
public abstract class AbstractRestClient implements RestClient {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Map<MediaType, MediaTypeFactory> _mediaTypeFacotries = new LinkedHashMap<>();

	private ExecutorService _executorService;

	private String _userAgent = getClass().getSimpleName() + "@" + Text.REFCODES_ORG.getText();

	private RestRequestHandler _handler = null;

	protected boolean _hasRequestCorrelation = true;

	protected boolean _hasSessionCorrelation = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new abstract rest client.
	 *
	 * @param aExecutorService the a executor service
	 */
	public AbstractRestClient( ExecutorService aExecutorService ) {
		_executorService = (aExecutorService != null) ? aExecutorService : ControlFlowUtility.createDaemonExecutorService();
		initMedaTypeFactories();
	}

	/**
	 * Instantiates a new abstract rest client.
	 */
	public AbstractRestClient() {
		this( ControlFlowUtility.createExecutorService() );
	}

	/**
	 * Adds the default {@link MediaTypeFactory} instances. Can be overridden.
	 */
	protected void initMedaTypeFactories() {
		addMediaTypeFactory( new ApplicationJsonFactory() );
		addMediaTypeFactory( new ApplicationXmlFactory() );
		addMediaTypeFactory( new TextPlainFactory() );
		addMediaTypeFactory( new ApplicationFormFactory() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void setRequestCorrelation( boolean hasRequestCorrelation ) {
		_hasRequestCorrelation = hasRequestCorrelation;
	}

	@Override
	public boolean hasRequestCorrelation() {
		return _hasRequestCorrelation;
	}

	@Override
	public void setSessionCorrelation( boolean hasSessionCorrelation ) {
		_hasSessionCorrelation = hasSessionCorrelation;
	}

	@Override
	public boolean hasSessionCorrelation() {
		return _hasSessionCorrelation;
	}

	@Override
	public RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		RestRequestHandler theHandler = _handler;
		if ( theHandler == null ) {
			throw new IllegalStateException( "Unable to process your request with HTTP-Method <" + aHttpMethod + "> for locator <" + aLocator + "> and query fields <" + new VerboseTextBuilderImpl().withElements( aQueryFields ).toString() + "> as no <" + RestRequestHandler.class.getSimpleName() + "> has been registered via <doRestRequest(...)>, aborting!" );
		}
		return theHandler.onRestRequest( new HttpClientRequestImpl( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest, this ) );
	}

	@Override
	public RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		RestCallerBuilder theRestCaller = new RestCallerBuilderImpl( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver, this );
		theRestCaller.getHeaderFields().putUserAgent( _userAgent );
		_executorService.execute( new RequestDaemon( theRestCaller ) );
		return theRestCaller;
	}

	@Override
	public RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return new RestRequestBuilderImpl( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest, this );
	}

	@Override
	public synchronized boolean addMediaTypeFactory( MediaTypeFactory aMediaTypeFactory ) {
		if ( _mediaTypeFacotries.containsKey( aMediaTypeFactory.getMediaType() ) ) return false;
		_mediaTypeFacotries.put( aMediaTypeFactory.getMediaType(), aMediaTypeFactory );
		return true;
	}

	@Override
	public MediaTypeFactory toMediaTypeFactory( MediaType aMediaType ) {
		return _mediaTypeFacotries.get( aMediaType );
	}

	@Override
	public MediaType[] getFactoryMediaTypes() {
		return _mediaTypeFacotries.keySet().toArray( new MediaType[_mediaTypeFacotries.size()] );
	}

	@Override
	public String getUserAgent() {
		return _userAgent;
	}

	@Override
	public void setUserAgent( String aUserAgent ) {
		_userAgent = aUserAgent;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sets the hook receiving a prepared {@link RestCallerBuilder} instance to
	 * be used to do the technical request with the technology chosen by the
	 * implementing sub-class.
	 * 
	 * @param aHandler The {@link RestRequestHandler} for handling the HTTP
	 *        request.
	 */
	protected void onRestRequest( RestRequestHandler aHandler ) {
		_handler = aHandler;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Do request correlation.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 */
	protected void doRequestCorrelation( RequestHeaderFields aRequestHeaderFields ) {
		if ( _hasRequestCorrelation ) {
			String theRequestId = Correlation.REQUEST.pullId();
			aRequestHeaderFields.putRequestId( theRequestId );
		}
	}

	/**
	 * Do session correlation.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 */
	protected void doSessionCorrelation( RequestHeaderFields aRequestHeaderFields ) {
		if ( _hasSessionCorrelation ) {
			String theSessionId = Correlation.SESSION.pullId();
			aRequestHeaderFields.putSessionId( theSessionId );
		}
	}

	/**
	 * Do request correlation.
	 *
	 * @param aResponseHeaderFields the a response Header-Fields
	 */
	protected void doRequestCorrelation( ResponseHeaderFields aResponseHeaderFields ) {
		if ( _hasRequestCorrelation ) {
			String theRequestId = aResponseHeaderFields.getRequestId();
			if ( theRequestId != null ) {
				Correlation.REQUEST.setId( theRequestId );
			}
		}
	}

	/**
	 * Do session correlation.
	 *
	 * @param aResponseHeaderFields the a response Header-Fields
	 */
	protected void doSessionCorrelation( ResponseHeaderFields aResponseHeaderFields ) {
		if ( _hasSessionCorrelation ) {
			String theSessionId = aResponseHeaderFields.getSessionId();
			if ( theSessionId != null ) {
				Correlation.SESSION.setId( theSessionId );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class RequestDaemon.
	 */
	private class RequestDaemon implements Runnable {

		private RestCallerBuilder _restCaller;

		/**
		 * Instantiates a new request daemon.
		 *
		 * @param aRestCallerBuilder the a rest caller builder
		 */
		public RequestDaemon( RestCallerBuilder aRestCallerBuilder ) {
			_restCaller = aRestCallerBuilder;
		}

		/**
		 * Run.
		 */
		@Override
		public void run() {
			RestRequestHandler theHandler = _handler;
			if ( theHandler == null ) {
				LOGGER.error( "Unable to process your request <" + _restCaller.toString() + "> as no <" + RestRequestHandler.class.getSimpleName() + "> has been registered via <doRestRequest(...)>, aborting!" );
				return;
			}
			try {
				// |--> Wait till the connection is opened / closed:
				synchronized ( _restCaller ) {
					while ( _restCaller.getConnectionStatus() == ConnectionStatus.NONE ) {
						try {
							_restCaller.wait( DaemonLoopSleepTime.MAX.getMilliseconds() );
						}
						catch ( InterruptedException e ) {}
						if ( _restCaller.getConnectionStatus() == ConnectionStatus.NONE ) {
							LOGGER.warn( "Your caller's <" + _restCaller + "> connection status is still <" + _restCaller.getConnectionStatus() + "> after <" + DaemonLoopSleepTime.MAX.getMilliseconds() + "> ms, execution of your request starts not earlyer than you calling the #open() method." );
						}
					}
				}
				// Wait till the connection is opened / closed <--|

				if ( _restCaller.getConnectionStatus() != ConnectionStatus.OPENED ) return;
				RestResponse theResponse = theHandler.onRestRequest( _restCaller );
				_restCaller.onResponse( new RestResponseEventImpl( theResponse, AbstractRestClient.this ) );
			}
			catch ( HttpResponseException e ) {
				LOGGER.error( ExceptionUtility.toMessage( e ), e );
				throw new HiddenException( e );
			}
		}
	}
}
