// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.junit.Test;
import org.refcodes.data.License;
import org.refcodes.io.impls.InputStreamStringBuilderImpl;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthFailureException;
import org.refcodes.net.BasicAuthRequiredException;
import org.refcodes.net.HeaderFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.RequestCookie;
import org.refcodes.net.impls.PortManagerSingleton;
import org.refcodes.rest.HttpRestClient;
import org.refcodes.rest.HttpRestServer;
import org.refcodes.rest.RestCallerBuilder;

/**
 * The Class HttpRestServerTest.
 */
public class HttpRestServerTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String LOCATOR = "/bla";

	private static final String BASE_URL = "http://localhost";

	private static final String BASE_LOCATOR = "/refcodes";

	private static final String COOKIE_A_NAME = "refcodes";

	private static final String COOKIE_A_VALUE = "org";

	private static final String COOKIE_A_VALUE_2 = "com";

	private static final String COOKIE_B_NAME = "funcodes";

	private static final String COOKIE_B_VALUE = "forever";

	private static final String LAST_NAME = "Bushnell";

	private static final String FIRST_NAME = "Nolan";

	private static final String USER_NAME = "nobody";

	private static final String PASSWORD = "secret";

	private static final String KEY_LAST_NAME = "lastName";

	private static final String KEY_FIRST_NAME = "firstName";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test http rest server.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServer() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test http rest server all methods 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServerAllMethods1() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onRequest( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test http rest server all methods 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServerAllMethods2() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onRequest( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = new Person( aRequest.getQueryFields().getFirst( KEY_FIRST_NAME ), aRequest.getQueryFields().getFirst( KEY_LAST_NAME ) );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.GET, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} );

		theCaller.getQueryFields().withPut( KEY_LAST_NAME, LAST_NAME ).withPut( KEY_FIRST_NAME, FIRST_NAME );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test http rest server stream A.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServerStreamA() throws IOException {
		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			InputStream theInputStream = aRequest.getRequest( InputStream.class );
			String theHttpBody = new InputStreamStringBuilderImpl().withInputStream( theInputStream ).toString();
			assertEquals( License.REFCODES_LICENSE.getText(), theHttpBody );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new ByteArrayInputStream( License.REFCODES_LICENSE.getText().getBytes( StandardCharsets.UTF_8 ) ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Line-Feed chars are preserved:.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServerStreamB() throws IOException {
		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			InputStream theInputStream = aRequest.getRequest( InputStream.class );
			String theHttpBody = new InputStreamStringBuilderImpl().withInputStream( theInputStream ).toString();
			assertNotEquals( License.REFCODES_LICENSE.getText(), theHttpBody );
			assertEquals( License.REFCODES_LICENSE.getText().replaceAll( "\\n", "\r\n" ), theHttpBody );

		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new ByteArrayInputStream( License.REFCODES_LICENSE.getText().replaceAll( "\\n", "\r\n" ).getBytes( StandardCharsets.UTF_8 ) ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Last Line-Feed char is preserved:.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestServerStreamC() throws IOException {
		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			InputStream theInputStream = aRequest.getRequest( InputStream.class );
			String theHttpBody = new InputStreamStringBuilderImpl().withInputStream( theInputStream ).toString();
			assertNotEquals( License.REFCODES_LICENSE.getText(), theHttpBody );
			assertEquals( (License.REFCODES_LICENSE.getText() + "\n"), theHttpBody );

		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new ByteArrayInputStream( (License.REFCODES_LICENSE.getText() + "\n").getBytes( StandardCharsets.UTF_8 ) ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test server cookie.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testServerCookie() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			RequestCookie theCookie = aRequest.getHeaderFields().getFirstCookie( COOKIE_A_NAME );
			assertEquals( COOKIE_A_NAME, theCookie.getKey() );
			assertEquals( COOKIE_A_VALUE, theCookie.getValue() );
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theBuilder = theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) );
		theBuilder.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		theBuilder.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test server cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testServerCookies() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {

			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );

			List<RequestCookie> theCookies = aRequest.getHeaderFields().getAllCookies();
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getValue().equals( COOKIE_A_VALUE_2 ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_A_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_A_VALUE_2, theCookies.get( b ).getValue() );

		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theBuilder = theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) );
		theBuilder.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		theBuilder.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE_2 );
		theBuilder.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test all server cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testAllServerCookies() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {

			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );

			List<RequestCookie> theCookies = aRequest.getHeaderFields().getAllCookies();
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getKey().equals( COOKIE_B_NAME ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_B_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_B_VALUE, theCookies.get( b ).getValue() );

		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theBuilder = theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) );
		theBuilder.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		theBuilder.getHeaderFields().addCookie( COOKIE_B_NAME, COOKIE_B_VALUE );
		theBuilder.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test basic auth success.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testBasicAuthSuccess() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onGet( LOCATOR, ( aRequest, aResponse ) -> {
			BasicAuthCredentials theCredentials = aRequest.getHeaderFields().getBasicAuthCredentials();
			LOGGER.debug( theCredentials.toString() + " := " + theCredentials.toHttpAuthorization() );
			assertEquals( USER_NAME, theCredentials.getUserName() );
			assertEquals( PASSWORD, theCredentials.getSecret() );
			aResponse.setResponse( "Authentification success!" );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.GET, LOCATOR, ( aResponse ) -> {
			assertEquals( HttpStatusCode.OK, aResponse.getHttpStatusCode() );
			String theMessage = aResponse.getResponse( String.class );
			LOGGER.debug( theMessage );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} );
		theCaller.getHeaderFields().putBasicAuthCredentials( USER_NAME, PASSWORD );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test basic auth failure.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testBasicAuthFailure() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onGet( LOCATOR, ( aRequest, aResponse ) -> {
			BasicAuthCredentials theCredentials = aRequest.getHeaderFields().getBasicAuthCredentials();
			LOGGER.debug( theCredentials.toString() + " := " + theCredentials.toHttpAuthorization() );
			assertEquals( USER_NAME, theCredentials.getUserName() );
			assertEquals( PASSWORD, theCredentials.getSecret() );
			throw new BasicAuthFailureException( "Authentification failed!" );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.GET, LOCATOR, ( aResponse ) -> {
			assertEquals( HttpStatusCode.UNAUTHORIZED, aResponse.getHttpStatusCode() );
			String theAuth = aResponse.getHeaderFields().getAuthenticate();
			LOGGER.debug( "Ahthenticate := " + theAuth );
			String theMessage = aResponse.getHttpBody();
			LOGGER.debug( "Error message := " + theMessage );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} );
		theCaller.getHeaderFields().putBasicAuthCredentials( USER_NAME, PASSWORD );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test basic auth required.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testBasicAuthRequired() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onGet( LOCATOR, ( aRequest, aResponse ) -> {
			BasicAuthCredentials theCredentials = aRequest.getHeaderFields().getBasicAuthCredentials();
			LOGGER.debug( theCredentials.toString() + " := " + theCredentials.toHttpAuthorization() );
			assertEquals( USER_NAME, theCredentials.getUserName() );
			assertEquals( PASSWORD, theCredentials.getSecret() );
			throw new BasicAuthRequiredException( "Authentification failed!" );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.GET, LOCATOR, ( aResponse ) -> {
			assertEquals( HttpStatusCode.UNAUTHORIZED, aResponse.getHttpStatusCode() );
			String theAuth = aResponse.getHeaderFields().getAuthenticate();
			LOGGER.debug( "Ahthenticate := " + theAuth );
			assertNotNull( theAuth );
			assertTrue( theAuth.startsWith( HeaderFields.BASIC_REALM ) );
			String theMessage = aResponse.getHttpBody();
			LOGGER.debug( "Error message := " + theMessage );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} );
		theCaller.getHeaderFields().putBasicAuthCredentials( USER_NAME, PASSWORD );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test query fields.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testQueryFields() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			assertEquals( FIRST_NAME, aRequest.getQueryFields().getFirst( KEY_FIRST_NAME ) );
			assertEquals( LAST_NAME, aRequest.getQueryFields().getFirst( KEY_LAST_NAME ) );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} );
		theCaller.getQueryFields().withAddTo( KEY_FIRST_NAME, FIRST_NAME ).withAddTo( KEY_LAST_NAME, LAST_NAME );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To base url.
	 *
	 * @param thePort the the port
	 * @return the string
	 */
	private String toBaseUrl( Integer thePort ) {
		return BASE_URL + ":" + thePort + BASE_LOCATOR;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class Person.
	 */
	public static class Person {

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return "Person [firstName=" + firstName + ", lastName=" + lastName + "]";
		}

		private String firstName;
		private String lastName;

		/**
		 * Instantiates a new person.
		 */
		public Person() {};

		/**
		 * Instantiates a new person.
		 *
		 * @param firstName the first name
		 * @param lastName the last name
		 */
		public Person( String firstName, String lastName ) {
			this.firstName = firstName;
			this.lastName = lastName;
		}

		/**
		 * Gets the first name.
		 *
		 * @return the first name
		 */
		public String getFirstName() {
			return firstName;
		}

		/**
		 * Sets the first name.
		 *
		 * @param firstName the new first name
		 */
		public void setFirstName( String firstName ) {
			this.firstName = firstName;
		}

		/**
		 * Gets the last name.
		 *
		 * @return the last name
		 */
		public String getLastName() {
			return lastName;
		}

		/**
		 * Sets the last name.
		 *
		 * @param lastName the new last name
		 */
		public void setLastName( String lastName ) {
			this.lastName = lastName;
		}
	}
}
