// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import static org.junit.Assert.*;

import java.io.IOException;
import java.util.List;

import org.junit.Test;
import org.refcodes.exception.MarshalException;
import org.refcodes.exception.UnmarshalException;
import org.refcodes.net.BadResponseException;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.RequestCookie;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.RequestHeaderFieldsImpl;
import org.refcodes.rest.LoopbackRestServer;

/**
 * The Class LoopbackRestServerTest.
 */
public class LoopbackRestServerTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String LOCATOR = "/bla";

	private static final String BASE_LOCATOR = "/refcodes";

	private static final String COOKIE_A_NAME = "refcodes";

	private static final String COOKIE_A_VALUE = "org";

	private static final String COOKIE_A_VALUE_2 = "com";

	private static final String COOKIE_B_NAME = "funcodes";

	private static final String COOKIE_B_VALUE = "forever";

	private static final String LAST_NAME = "Bushnell";

	private static final String FIRST_NAME = "Nolan";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test loopback rest server.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpStatusException the http status exception
	 * @throws MarshalException the marshal exception
	 * @throws BadResponseException the bad response exception
	 * @throws UnmarshalException the unmarshal exception
	 */
	@Test
	public void testLoopbackRestServer() throws IOException, HttpStatusException, MarshalException, BadResponseException, UnmarshalException {
		LoopbackRestServer theRestServer = new LoopbackRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();

		MediaTypeFactory theFactory = new ApplicationJsonFactory();
		RequestHeaderFields theHeaderFields = new RequestHeaderFieldsImpl().withAcceptTypes( MediaType.APPLICATION_JSON );
		HttpServerResponse theResponse = theRestServer.onHttpRequest( null, null, HttpMethod.POST, BASE_LOCATOR + LOCATOR, null, theHeaderFields, theFactory.fromUnmarshaled( new Person( FIRST_NAME, LAST_NAME ) ) );
		Person thePerson = theFactory.toUnmarshaled( theResponse.toHttpBody(), Person.class );
		assertEquals( FIRST_NAME, thePerson.getFirstName() );
		assertEquals( LAST_NAME, thePerson.getLastName() );
	}

	/**
	 * Test server cookie.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpStatusException the http status exception
	 * @throws MarshalException the marshal exception
	 * @throws BadResponseException the bad response exception
	 * @throws UnmarshalException the unmarshal exception
	 */
	@Test
	public void testServerCookie() throws IOException, HttpStatusException, MarshalException, BadResponseException, UnmarshalException {
		LoopbackRestServer theRestServer = new LoopbackRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			RequestCookie theCookie = aRequest.getHeaderFields().getFirstCookie( COOKIE_A_NAME );
			assertEquals( COOKIE_A_NAME, theCookie.getKey() );
			assertEquals( COOKIE_A_VALUE, theCookie.getValue() );
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();

		MediaTypeFactory theFactory = new ApplicationJsonFactory();
		RequestHeaderFields theHeaderFields = new RequestHeaderFieldsImpl().withAcceptTypes( MediaType.APPLICATION_JSON );
		theHeaderFields.addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		HttpServerResponse theResponse = theRestServer.onHttpRequest( null, null, HttpMethod.POST, BASE_LOCATOR + LOCATOR, null, theHeaderFields, theFactory.fromUnmarshaled( new Person( FIRST_NAME, LAST_NAME ) ) );
		Person thePerson = theFactory.toUnmarshaled( theResponse.toHttpBody(), Person.class );
		assertEquals( FIRST_NAME, thePerson.getFirstName() );
		assertEquals( LAST_NAME, thePerson.getLastName() );
	}

	/**
	 * Test server cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws BadResponseException the bad response exception
	 * @throws UnmarshalException the unmarshal exception
	 * @throws HttpStatusException the http status exception
	 * @throws MarshalException the marshal exception
	 */
	@Test
	public void testServerCookies() throws IOException, BadResponseException, UnmarshalException, HttpStatusException, MarshalException {
		LoopbackRestServer theRestServer = new LoopbackRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {

			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );

			List<RequestCookie> theCookies = aRequest.getHeaderFields().getAllCookies();
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getValue().equals( COOKIE_A_VALUE_2 ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_A_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_A_VALUE_2, theCookies.get( b ).getValue() );

		} ).open();

		MediaTypeFactory theFactory = new ApplicationJsonFactory();
		RequestHeaderFields theHeaderFields = new RequestHeaderFieldsImpl().withAcceptTypes( MediaType.APPLICATION_JSON );
		theHeaderFields.addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		theHeaderFields.addCookie( COOKIE_A_NAME, COOKIE_A_VALUE_2 );
		HttpServerResponse theResponse = theRestServer.onHttpRequest( null, null, HttpMethod.POST, BASE_LOCATOR + LOCATOR, null, theHeaderFields, theFactory.fromUnmarshaled( new Person( FIRST_NAME, LAST_NAME ) ) );
		Person thePerson = theFactory.toUnmarshaled( theResponse.toHttpBody(), Person.class );
		assertEquals( FIRST_NAME, thePerson.getFirstName() );
		assertEquals( LAST_NAME, thePerson.getLastName() );
	}

	/**
	 * Test all server cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws BadResponseException the bad response exception
	 * @throws UnmarshalException the unmarshal exception
	 * @throws HttpStatusException the http status exception
	 * @throws MarshalException the marshal exception
	 */
	@Test
	public void testAllServerCookies() throws IOException, BadResponseException, UnmarshalException, HttpStatusException, MarshalException {
		LoopbackRestServer theRestServer = new LoopbackRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {

			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );

			List<RequestCookie> theCookies = aRequest.getHeaderFields().getAllCookies();
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getKey().equals( COOKIE_B_NAME ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_B_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_B_VALUE, theCookies.get( b ).getValue() );

		} ).open();

		MediaTypeFactory theFactory = new ApplicationJsonFactory();
		RequestHeaderFields theHeaderFields = new RequestHeaderFieldsImpl().withAcceptTypes( MediaType.APPLICATION_JSON );
		theHeaderFields.addCookie( COOKIE_A_NAME, COOKIE_A_VALUE );
		theHeaderFields.addCookie( COOKIE_B_NAME, COOKIE_B_VALUE );
		HttpServerResponse theResponse = theRestServer.onHttpRequest( null, null, HttpMethod.POST, BASE_LOCATOR + LOCATOR, null, theHeaderFields, theFactory.toMarshaled( new Person( FIRST_NAME, LAST_NAME ) ) );
		Person thePerson = theFactory.toUnmarshaled( theResponse.toHttpBody(), Person.class );
		assertEquals( FIRST_NAME, thePerson.getFirstName() );
		assertEquals( LAST_NAME, thePerson.getLastName() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class Person.
	 */
	public static class Person {

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return "Person [firstName=" + firstName + ", lastName=" + lastName + "]";
		}

		private String firstName;
		private String lastName;

		/**
		 * Instantiates a new person.
		 */
		public Person() {};

		/**
		 * Instantiates a new person.
		 *
		 * @param firstName the first name
		 * @param lastName the last name
		 */
		public Person( String firstName, String lastName ) {
			this.firstName = firstName;
			this.lastName = lastName;
		}

		/**
		 * Gets the first name.
		 *
		 * @return the first name
		 */
		public String getFirstName() {
			return firstName;
		}

		/**
		 * Sets the first name.
		 *
		 * @param firstName the new first name
		 */
		public void setFirstName( String firstName ) {
			this.firstName = firstName;
		}

		/**
		 * Gets the last name.
		 *
		 * @return the last name
		 */
		public String getLastName() {
			return lastName;
		}

		/**
		 * Sets the last name.
		 *
		 * @param lastName the new last name
		 */
		public void setLastName( String lastName ) {
			this.lastName = lastName;
		}
	}
}
