// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.io.InputStream;
import java.net.InetSocketAddress;

import org.refcodes.matcher.WildcardSubstitutes;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerRequestImpl;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;

/**
 * Implementation of the {@link RestRequestEvent} interface as good old POJO for
 * use by different {@link RestServer} implementations.
 */
public class RestRequestEventImpl extends HttpServerRequestImpl implements RestRequestEvent {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private InetSocketAddress _remoteAddress;

	private InetSocketAddress _localAddress;

	private WildcardSubstitutes _wildcardSubstitutes;

	private String _realm;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link RestRequestEvent} with all required attributes.
	 * 
	 * @param aLocalAddress The local address where the event is being received.
	 * @param aRemoteAddress The remote address from which the request
	 *        originates.
	 * @param aHttpMethod The {@link HttpMethod} with which the request has been
	 *        sent.
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aWildcardSubstitutes The text being substituted by the wildcard of
	 *        the {@link RestEndpoint}'s Locator-Pattern.
	 * @param aHeaderFields The {@link RequestHeaderFields} sent by the request.
	 * @param aHttpInputStream The {@link InputStream} representing the
	 *        request's HTTP body.
	 * @param aRestServer The system firing the event.
	 */
	public RestRequestEventImpl( InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, Url aUrl, WildcardSubstitutes aWildcardSubstitutes, RequestHeaderFields aHeaderFields, InputStream aHttpInputStream, RestServer aRestServer ) {
		super( aHttpMethod, aUrl, aHeaderFields, aHttpInputStream, aRestServer );
		_wildcardSubstitutes = aWildcardSubstitutes;
		_remoteAddress = aRemoteAddress;
		_localAddress = aLocalAddress;
		_realm = aRestServer.getRealm();
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RestServer getSource() {
		return (RestServer) _mediaTypeFactoryLookup;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacements() {
		return _wildcardSubstitutes.getWildcardReplacements();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getWildcardReplacementAt( int aIndex ) {
		return _wildcardSubstitutes.getWildcardReplacementAt( aIndex );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getWildcardReplacement( String aWildcardName ) {
		return _wildcardSubstitutes.getWildcardReplacement( aWildcardName );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacements( String... aWildcardNames ) {
		return _wildcardSubstitutes.getWildcardReplacements( aWildcardNames );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacementsAt( int... aIndexes ) {
		return _wildcardSubstitutes.getWildcardReplacementsAt( aIndexes );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InetSocketAddress getRemoteAddress() {
		return _remoteAddress;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InetSocketAddress getLocalAddress() {
		return _localAddress;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getRealm() {
		return _realm;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
