// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.InetSocketAddress;

import org.refcodes.matcher.WildcardSubstitutes;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthRequestImpl;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.Url;

/**
 * The {@link BasicAuthEventImpl} class implements the {@link BasicAuthEvent}
 * type.
 *
 */
public class BasicAuthEventImpl extends BasicAuthRequestImpl implements BasicAuthEvent {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private InetSocketAddress _localAddress;
	private InetSocketAddress _remoteAddress;
	private RestServer _restServer;
	private WildcardSubstitutes _wildcardSubstitutes;
	private String _realm;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * @param aLocalAddress The local address where the event is being received.
	 * @param aRemoteAddress The remote address from which the request
	 *        originates.
	 * @param aHttpMethod The {@link HttpMethod} with which the request has been
	 *        sent.
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aWildcardSubstitutes The text being substituted by the wildcard of
	 *        the {@link RestEndpoint}'s Locator-Pattern.
	 * @param aCredentials The credentials retrieved from the request.
	 * @param aRealm The realm to which this request is being addressed.
	 * @param aRestServer The system firing the event.
	 */
	public BasicAuthEventImpl( InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, Url aUrl, WildcardSubstitutes aWildcardSubstitutes, BasicAuthCredentials aCredentials, String aRealm, RestServer aRestServer ) {
		super( aHttpMethod, aUrl, aCredentials );
		_restServer = aRestServer;
		_localAddress = aLocalAddress;
		_remoteAddress = aRemoteAddress;
		_wildcardSubstitutes = aWildcardSubstitutes;
		_realm = aRealm;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacements() {
		return _wildcardSubstitutes.getWildcardReplacements();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getWildcardReplacementAt( int aIndex ) {
		return _wildcardSubstitutes.getWildcardReplacementAt( aIndex );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getWildcardReplacement( String aWildcardName ) {
		return _wildcardSubstitutes.getWildcardReplacement( aWildcardName );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacements( String... aWildcardNames ) {
		return _wildcardSubstitutes.getWildcardReplacements( aWildcardNames );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] getWildcardReplacementsAt( int... aIndexes ) {
		return _wildcardSubstitutes.getWildcardReplacementsAt( aIndexes );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InetSocketAddress getRemoteAddress() {
		return _remoteAddress;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InetSocketAddress getLocalAddress() {
		return _localAddress;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getRealm() {
		return _realm;
	}

	@Override
	public RestServer getSource() {
		return _restServer;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
