// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.File;
import java.util.Arrays;

import org.refcodes.data.DelimeterConsts;
import org.refcodes.data.SystemConsts;
import org.refcodes.exception.ExceptionUtility;

/**
 * Utility class with helpful methods for working with a system's environment
 * variables.
 */
public final class EnvironmentUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private EnvironmentUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////
	/**
	 * This method converts a relative path to an absolute path relative to the
	 * user's home folder. In case there is no user home set in the system's
	 * environment, then an illegal state exception is thrown. If the path is an
	 * absolute path, then no modifications are applied to the absolute path and
	 * it is returned unmodified.
	 * 
	 * -------------------------------------------------------------------------
	 * ATTENTION: In case the provided path already points to an existing file,
	 * then the absolute path for that path is returned!
	 * -------------------------------------------------------------------------
	 * 
	 * @param aPath The path to be converted to an absolute path in case it was
	 *        a relative path.
	 * @return An absolute path, either the given path in case it was an
	 *         absolute path or a path relative to the user's home folder.
	 * 
	 * @throws IllegalStateException in case the user's home folder is not
	 *         defined and a relative path has been provided.
	 */
	public static String toAbsoluteUserHomePath( String aPath ) throws IllegalStateException {

		// -----------------------------------------------
		// Does the path specify already an existing file?
		// -----------------------------------------------
		File theConfigFile = new File( aPath );
		if ( theConfigFile.exists() ) {
			return theConfigFile.getAbsolutePath();
		}
		// -----------------------------------------------

		try {
			return toAbsolutePathFromEnvironment( aPath, SystemConsts.ENV_VAR_USER_HOME );
		}
		catch ( IllegalArgumentException e ) {
			throw new IllegalStateException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * This method converts a relative path to an absolute path relative to the
	 * path specified in the first system property provided being set with a
	 * value. In case the path provided is an absolute path, then the path is
	 * returned unmodified. case there is no such system property set in the
	 * system's environment or the value is empty, then an illegal argument
	 * exception is thrown. If the path is an absolute path, then no
	 * modifications are applied to the absolute path and it is returned
	 * unmodified.
	 * 
	 * -------------------------------------------------------------------------
	 * ATTENTION: In case the provided path already points to an existing file,
	 * then the absolute path for that path is returned!
	 * -------------------------------------------------------------------------
	 * 
	 * @param aPath The path to be converted to an absolute path in case it was
	 *        a relative path.
	 * @param aEnvHomePropertyNames The names of the system properties which's
	 *        value is to be used in order to resolve the path.
	 * 
	 * @return An absolute path, either the given path in case it was an
	 *         absolute path or a path relative to the specified system's
	 *         property value (a folder).
	 * 
	 * @throws IllegalArgumentException in case the system property name is not
	 *         defined or is empty and a relative path has been provided.
	 */
	public static String toAbsolutePathFromEnvironment( String aPath, String... aEnvHomePropertyNames ) throws IllegalArgumentException {

		if ( (aPath == null) || (aPath.length() == 0) || (aPath.charAt( 0 ) == DelimeterConsts.PATH_DELIMETER) || ((aPath.length() > 1) && (aPath.charAt( 1 ) == ':')) ) {
			return aPath;
		}

		// -----------------------------------------------
		// Does the path specify already an existing file?
		// -----------------------------------------------
		File theConfigFile = new File( aPath );
		if ( theConfigFile.exists() ) {
			return theConfigFile.getAbsolutePath();
		}
		// -----------------------------------------------

		String eHomePath = null;
		for ( int i = 0; i < aEnvHomePropertyNames.length; i++ ) {
			eHomePath = System.getenv( aEnvHomePropertyNames[i] );
			// LOGGER.debug( "The system environment variable \"" +
			// aEnvHomePropertyNames[i] + "\" is set to \"" + eHomePath + "\"."
			// );
			if ( (eHomePath != null) && (eHomePath.length() > 0) ) {
				String thePath = eHomePath + DelimeterConsts.PATH_DELIMETER + aPath;
				// LOGGER.debug( "Assembled absolute path \"" + thePath +
				// "\" from the env variable \"" + aEnvHomePropertyNames[i] +
				// "\" with value \"" + eHomePath +
				// "\" and the provided path \"" + aPath + "\"." );
				return thePath;
			}

		}
		throw new IllegalArgumentException( "A relative path \"" + aPath + "\" was provided though the system environment variable(s) " + Arrays.asList( aEnvHomePropertyNames ).toString() + " is not set, unable to create an absolute path. Either set the system environment variable or specify an absolute path." );
	}

	/**
	 * This method concerts the given path to a platform specific path, mainly
	 * replacing the path separator accordingly.
	 * 
	 * @param aPath The path to be converted.
	 * @return The converted path.
	 */
	public static String toPlatformSpecificPath( String aPath ) {
		String theFileSeparator = System.getProperty( SystemConsts.SYS_PROP_FILE_DELIMETER );
		if ( theFileSeparator != null && theFileSeparator.length() == 1 ) {
// @formatter:off
				/*
					char theFileSeapratorChar = theFileSeparator.charAt( 0 );
					String thePath = aPath.replace( RefcodesConstants.DEFAULT_PATH_DELIMETER, theFileSeapratorChar );
					LOGGER.debug( "Converted to a platform specifc path \"" + thePath + "\" from the provided path \"" + aPath + "\"." );
				*/
			// @formatter:on
		}

		else {
// @formatter:off	
				/*				
					LOGGER.warn( "Left \"" + aPath + "\" unchanged as no (or an invalid) path separator was determined with the env property \"" + RefcodesConstants.ENV_FILE_DELIMETER + "\" from the system environment." );				
				*/
			// @formatter:on
		}
		return aPath;
	}

}
