// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.management.ManagementFactory;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.refcodes.collection.Properties;
import org.refcodes.data.DelimeterConsts;
import org.refcodes.data.FileSystemConsts;
import org.refcodes.data.ReflectionConsts;
import org.refcodes.data.RuntimeConsts;
import org.refcodes.data.SystemConsts;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;
import org.refcodes.runtime.impls.RuntimePropertiesImpl;

/**
 * Utility for acquiring runtime information on software systems, classes or
 * objects.
 */
public final class RuntimeUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private RuntimeUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the stack trace for the current thread.
	 * 
	 * @return The stack trace from the current thread.
	 */
	public static String toStackTrace() {
		StringWriter theStringWriter = new StringWriter();
		new Throwable( "" ).printStackTrace( new PrintWriter( theStringWriter ) );
		return theStringWriter.toString();
	}

	/**
	 * Returns the stack trace element belonging to the direct caller of this
	 * method. When you use this method in your code, you get the stack trace
	 * element of your method (invoking this method).
	 * 
	 * @return The stack element of the direct caller of this method.
	 */
	public static StackTraceElement getCallerStackTraceElement() {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( !isSkipStackTraceElement( eStackTraceElement ) ) { return eStackTraceElement; }
		}
		return null;
	}

	/**
	 * Returns the stack trace element belonging to the caller of the callee.
	 * Best you do not use the {@link Object#getClass()} method, instead use
	 * YourClass.class as as of inheritance, {@link Object#getClass()} returns
	 * the wrong type not being the actual callee!
	 * 
	 * @param aCallee The callee class which wants to find out who called it.
	 * 
	 * @return The stack element of the caller of the callee or null if the
	 *         callee is not present or if there is no caller of the given
	 *         callee in the current stack trace.
	 */
	public static StackTraceElement getCallerStackTraceElement( Class<?> aCallee ) {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		String theCalleeClassName = aCallee.getName();
		boolean hasCalleeElement = false;
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( hasCalleeElement && !eStackTraceElement.getClassName().equals( theCalleeClassName ) ) { return eStackTraceElement; }
			if ( eStackTraceElement.getClassName().equals( theCalleeClassName ) ) {
				hasCalleeElement = true;
			}
		}
		return null;
	}

	/**
	 * Same as {@link #getCallerStackTraceElement(Class)} with the difference
	 * that the passed callees are tried out one after the other until the first
	 * caller determined for a callee is returned.
	 */
	public static StackTraceElement getCallerStackTraceElement( Class<?>... aCallees ) {
		StackTraceElement eCaller = null;
		for ( Class<?> eClallee : aCallees ) {
			eCaller = getCallerStackTraceElement( eClallee );
			if ( eCaller != null ) { return eCaller; }
		}
		return null;
	}

	/**
	 * Returns the stack trace element belonging to the caller of the callee.
	 * Best you do not use the {@link Object#getClass()} method, instead use
	 * YourClass.class as as of inheritance, {@link Object#getClass()} returns
	 * the wrong type not being the actual callee!
	 * 
	 * @param aCalleeClassName The callee class name which wants to find out who
	 *        called it.
	 * 
	 * @return The stack element of the caller of the callee or null if the
	 *         callee is not present or if there is no caller of the given
	 *         callee in the current stack trace.
	 */
	public static StackTraceElement getCallerStackTraceElement( String aCalleeClassName ) {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		boolean hasCalleeElement = false;
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( hasCalleeElement && !eStackTraceElement.getClassName().equals( aCalleeClassName ) ) { return eStackTraceElement; }
			if ( eStackTraceElement.getClassName().equals( aCalleeClassName ) ) {
				hasCalleeElement = true;
			}
		}
		return null;
	}

	/**
	 * Same as {@link #getCallerStackTraceElement(String)} with the difference
	 * that the passed callees are tried out one after the other until the first
	 * caller determined for a callee is returned.
	 */
	public static StackTraceElement getCallerStackTraceElement( String... aCalleeClassNames ) {
		StackTraceElement eCaller = null;
		for ( String eClallee : aCalleeClassNames ) {
			eCaller = getCallerStackTraceElement( eClallee );
			if ( eCaller != null ) { return eCaller; }
		}
		return null;
	}

	/**
	 * Returns the type of the ({@link Class}) belonging to the direct caller of
	 * this method. When you use this method in your code, you get the
	 * {@link Class} of your method (invoking this method).
	 * 
	 * @return The type ({@link Class}) of the direct caller of this method.
	 */
	public static Class<?> geCallerType() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return toClass( theStackTraceElement );
	}

	/**
	 * Returns the type of the ({@link Class}) belonging to the caller of the
	 * callee.
	 * 
	 * @param aCallee The callee class which wants to find out who called it.
	 * 
	 * @return The type ({@link Class}) of the caller of the caller of this
	 *         method.
	 */
	public static Class<?> getCallerType( Class<?> aCallee ) {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement( aCallee );
		return toClass( theStackTraceElement );
	}

	/**
	 * Same as {@link #getCallerType(Class)} with the difference that the passed
	 * callees are tried out one after the other until the first caller
	 * determined for a callee is returned.
	 */
	public static Class<?> getCallerType( Class<?>... aCallees ) {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement( aCallees );
		return toClass( theStackTraceElement );
	}

	/**
	 * Retrieves the {@link Class} type to which the {@link StackTraceElement}
	 * belongs.
	 * 
	 * @param aStackTraceElement The {@link StackTraceElement} for which to get
	 *        the according {@link Class}.
	 * 
	 * @return The type ({@link Class}) of the according
	 *         {@link StackTraceElement}.
	 */
	public static Class<?> toClass( StackTraceElement aStackTraceElement ) {
		try {
			return Class.forName( aStackTraceElement.getClassName() );
		}
		catch ( ClassNotFoundException e ) {
			throw new HiddenException( e );
		}
	}

	/**
	 * Retrieves the method name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the method name.
	 * 
	 * @return The method name or null in case the stack trace element was null.
	 */
	public static String toMethodName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getMethodName();
	}

	/**
	 * Returns the class name part from a stack trace element.
	 * 
	 * Retrieves the fully qualified class name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the class name.
	 * 
	 * @return The class name without the package declaration or null in case
	 *         the stack trace element was null.
	 */
	public static String toClassName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return toClassName( aStackTraceElement.getClassName() );
	}

	/**
	 * Retrieves the fully qualified class name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified class name.
	 * 
	 * @return The fully qualified class name or null in case the stack trace
	 *         element was null.
	 */
	public static String toFullyQualifiedClassName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getClassName();
	}

	/**
	 * Retrieves the fully qualified method name from a stack trace element.
	 * This adds the method name to the fully qualified path name separated by a
	 * hash "#".
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified method name.
	 * 
	 * @return The fully qualified method name or null in case the stack trace
	 *         element was null.
	 */
	public static String toFullyQualifiedMethodName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getClassName() + ReflectionConsts.METHOD_NAME_SEPARATOR + aStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toFullyQualifiedClassName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getClassName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toFullyQualifiedMethodName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getClassName() + ReflectionConsts.METHOD_NAME_SEPARATOR + theStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toMethodName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the class name of the caller of this method without the fully
	 * qualified package name part.
	 * 
	 * @return The class name.
	 */
	public static String toClassName() {
		return toClassName( toFullyQualifiedClassName() );
	}

	/**
	 * Retrieves the fully qualified package name of the caller of this method
	 * without the class name part.
	 * 
	 * @return The fully qualified package name.
	 */
	public static String toFullyQualifiedPackageName() {
		return toFullyQualifiedPackageName( toFullyQualifiedClassName() );
	}

	/**
	 * Retrieves the fully qualified package name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified package name.
	 * 
	 * @return The fully qualified package name.
	 */
	public static String toFullyQualifiedPackageName( StackTraceElement aStackTraceElement ) {
		return toFullyQualifiedPackageName( toFullyQualifiedClassName( aStackTraceElement ) );
	}

	/**
	 * Returns the class name part from a fully qualified class name (which has
	 * the fully qualified package name as part of its name).
	 * 
	 * @param aFullyQualifiedClassName The fully qualified class name.
	 * 
	 * @return The class name without the package declaration.
	 */
	public static String toClassName( String aFullyQualifiedClassName ) {
		String theClassName;
		int theClassIndex = aFullyQualifiedClassName.lastIndexOf( '.' );
		if ( theClassIndex != -1 ) {
			theClassName = aFullyQualifiedClassName.substring( theClassIndex + 1 );
		}
		else {
			theClassName = aFullyQualifiedClassName;
		}
		int theInnerClassIndex = theClassName.indexOf( '$' );
		if ( theInnerClassIndex != -1 ) {
			theClassName = theClassName.substring( theInnerClassIndex + 1 );
		}
		return theClassName;
	}

	/**
	 * Returns the fully qualified package name part from a fully qualified
	 * class name (which has the fully qualified package name as part of its
	 * name).
	 * 
	 * @param aFullyQualifiedClassName The fully qualified class name.
	 * 
	 * @return The fully qualified package name without the class name.
	 */
	public static String toFullyQualifiedPackageName( String aFullyQualifiedClassName ) {
		int theIndex = aFullyQualifiedClassName.lastIndexOf( '.' );
		if ( theIndex != -1 ) { return aFullyQualifiedClassName.substring( 0, theIndex ); }
		return "";
	}

	/**
	 * A {@link Cloneable} object cannot directly be cloned by casting it to be
	 * {@link Cloneable} :-( Thereforee this method does the job.
	 * 
	 * Citation From Josh Bloch's Effective Java: "The {@link Cloneable}
	 * interface was intended as a mixin interface for objects to advertise that
	 * they permit cloning. Unfortunately it fails to serve this purpose ...
	 * This is a highly atypical use of interfaces and not one to be emulated
	 * ... In order for implementing the interface to have any effect on a
	 * class, it and all of its superclasses must obey a fairly complex,
	 * unenforceable and largely undocumented protocol"
	 * 
	 * @see "http://stackoverflow.com/questions/1138769/why-is-the-clone-method-protected-in-java-lang-object"
	 * 
	 * @param aObj The object to be cloned.
	 * 
	 * @return The cloned object.
	 * 
	 * @throws CloneNotSupportedException in case the object cannot be cloned.
	 */

	@SuppressWarnings("unchecked")
	public static <T> T toClone( T aObj ) throws CloneNotSupportedException {
		if ( aObj instanceof Cloneable ) {
			try {
				return (T) aObj.getClass().getMethod( "clone" ).invoke( aObj );
			}
			catch ( IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException e ) {
				throw new CloneNotSupportedException( ExceptionUtility.toMessage( e ) );
			}
		}
		else {
			throw new CloneNotSupportedException( "The type \"" + aObj.getClass().getName() + "}\" does not implement the \"" + Cloneable.class.getName() + "\" interface." );
		}
	}

	/**
	 * Creates a string of a super class's {@link Object#toString()} method and
	 * the provided "toString" text.
	 * 
	 * @param aToString The provided "toString" text.
	 * 
	 * @param aSuperToString A super class's {@link Object#toString()} method's
	 *        {@link String}.
	 * 
	 * @return The "concatenated" and formatted new {@link String} to be
	 *         returned by an implementing class's {@link Object#toString()}
	 *         method.
	 */
	public static String toString( String aToString, String aSuperToString ) {
		return aToString + " (" + aSuperToString + ")";
	}

	/**
	 * Bad hack to get the JVM's (process ID) PID of the process running your
	 * JVM instance.
	 * 
	 * @return The PID (process ID) of the JVM running your thread.
	 * 
	 * @see "http://stackoverflow.com/questions/35842/how-can-a-java-program-get-its-own-process-id"
	 */
	public static Long getPid() {
		Long thePid = null;
		try {
			thePid = Long.parseLong( System.getProperty( SystemConsts.SYS_PROP_PID ) );
		}
		catch ( NumberFormatException e1 ) {
			String theJvmName = ManagementFactory.getRuntimeMXBean().getName();
			if ( theJvmName.indexOf( '@' ) != -1 ) {
				try {
					thePid = Long.parseLong( theJvmName.substring( 0, theJvmName.indexOf( '@' ) ) );
				}
				catch ( NumberFormatException e2 ) {
					// Unable to determine PID
				}
			}
		}
		return thePid;
	}

	/**
	 * Bad hack to kill an OS thread by PID. The current threads does not wait
	 * till the operation finished.
	 * 
	 * @param aPid The process ID (PID) of the process to kill.
	 * 
	 * @return The {@link Process} object representing the kill operation. This
	 *         instance will let you wait till the operation finished
	 *         {@link Process#waitFor()} and provides access to the
	 *         {@link Process#exitValue()}
	 * 
	 * @throws IOException Thrown in case of failing to successfully execute the
	 *         kill operation.
	 * 
	 * @see "http://stackoverflow.com/questions/9573696/kill-a-process-based-on-pid-in-java"
	 * @see "http://stackoverflow.com/questions/2950338/how-can-i-kill-a-linux-process-in-java-with-sigkill-process-destroy-does-sigte"
	 */
	public static Process killProcess( Long aPid ) throws IOException {
		Process theProcess = null;
		String theCmd;
		switch ( SystemUtility.getOperatingSystem() ) {
		case WINDOWS:
			theCmd = "taskkill /F /PID " + aPid;
			theProcess = Runtime.getRuntime().exec( theCmd );
			break;
		case UNIX:
			theCmd = "kill -9 " + aPid;
			theProcess = Runtime.getRuntime().exec( theCmd );
			break;
		default:
		}
		return theProcess;
	}

	/**
	 * Bad hack to kill an OS thread by PID. The current threads does wait till
	 * the operation finished.
	 * 
	 * @param aPid The process ID (PID) of the process to kill.
	 * 
	 * @return True in case killing the process was successful (e.g. the kill
	 *         operation returned an exit code 0), else false.
	 * 
	 * @throws IOException Thrown in case of failing to successfully execute the
	 *         kill operation.
	 * 
	 * @see "http://stackoverflow.com/questions/9573696/kill-a-process-based-on-pid-in-java"
	 * @see "http://stackoverflow.com/questions/2950338/how-can-i-kill-a-linux-process-in-java-with-sigkill-process-destroy-does-sigte"
	 */
	public static boolean kill( Long aPid ) throws IOException, InterruptedException {
		Process theProcess = killProcess( aPid );
		if ( theProcess != null ) {
			theProcess.waitFor();
			return (theProcess.exitValue() == 0);
		}
		return false;
	}

	/**
	 * Generates the base path relative to the given class location. Depending
	 * on the runtime, the path is truncated till the required path is
	 * determined.
	 *
	 * @return The base path of this application.
	 * @throws IOException thrown in case determining the path failed due to
	 *         I/O.
	 */
	public static File toAppBaseDir() throws IOException {
		Class<?> baseClass = getMainClass();
		if ( baseClass == null ) baseClass = RuntimeUtility.class;
		String basePath = baseClass.getProtectionDomain().getCodeSource().getLocation().getPath();
		if ( basePath.startsWith( FileSystemConsts.FILE_PROTOCOL_PREFIX ) ) {
			basePath = basePath.substring( FileSystemConsts.FILE_PROTOCOL_PREFIX.length() );
		}
		String truncate = baseClass.getCanonicalName().replace( '.', '/' );
		truncate = truncate.substring( 0, truncate.indexOf( baseClass.getSimpleName() ) );
		int endIndex = basePath.indexOf( truncate );
		if ( endIndex > 0 ) {
			basePath = basePath.substring( 0, endIndex );
		}
		if ( basePath.endsWith( "" + DelimeterConsts.PATH_DELIMETER ) ) {
			basePath = basePath.substring( 0, basePath.length() - 1 );
		}
		if ( basePath.endsWith( ReflectionConsts.CLASSES_FOLDER ) ) {
			basePath = basePath.substring( 0, basePath.indexOf( ReflectionConsts.CLASSES_FOLDER ) );
		}
		int index = basePath.toLowerCase().indexOf( FileSystemConsts.JAR_URL_RESOURCE_MARKER.toLowerCase() );
		if ( index != -1 ) {
			basePath = basePath.substring( 0, index );
			index = basePath.lastIndexOf( DelimeterConsts.PATH_DELIMETER );
			if ( index != -1 ) {
				basePath = basePath.substring( 0, index );
			}
		}
		if ( !basePath.endsWith( "" + DelimeterConsts.PATH_DELIMETER ) ) {
			basePath = basePath + DelimeterConsts.PATH_DELIMETER;
		}
		File baseFile = new File( basePath );
		if ( basePath.endsWith( ReflectionConsts.TARGET_FOLDER + DelimeterConsts.PATH_DELIMETER ) ) { return baseFile; }
		File parentFile = baseFile.getParentFile();
		return parentFile;
	}

	/**
	 * Returns the main class launching the application.
	 *
	 * @return The main class.
	 */
	@SuppressWarnings("rawtypes")
	public static Class getMainClass() {
		StackTraceElement[] stack = Thread.currentThread().getStackTrace();
		StackTraceElement eElement;

		Class<?> mainClass;
		// Try to find dab main class:
		for ( int i = stack.length - 1; i > 0; i-- ) {
			eElement = stack[i];
			if ( eElement.getMethodName().equals( ReflectionConsts.MAIN_METHOD ) ) {
				try {
					mainClass = Class.forName( eElement.getClassName() );
					if ( mainClass.getProtectionDomain().getCodeSource() != null ) { return mainClass; }
				}
				catch ( ClassNotFoundException e ) {}
			}
		}
		return null;
	}

	/**
	 * Returns a list of (existing) folders which are candidates for external
	 * resources. The applications base directory (where your JAR or your
	 * classes reside) is taken and a list of directories relative to this base
	 * directory is generated as defined in the
	 * {@link RuntimeConsts#CONFIG_DIR_NAMES}: The actual directories being
	 * returned (in case them exist) are as follows, relative to your
	 * applications base directory:
	 * 
	 * <ul>
	 * <li>../config"</li>
	 * <li>../etc"</li>
	 * <li>../settings"</li>
	 * <li>../.config"</li>
	 * <li>../settings"</li>
	 * <li>."</li>
	 * <li>./config"</li>
	 * <li>./etc"</li>
	 * <li>./settings"</li>
	 * <li>./.config"</li>
	 * <li>./settings"</li>
	 * 
	 * In case you pass a JVM argument via
	 * "-Dconfig.dir=path_to_your_config_dir" (where path_to_your_config_dir
	 * stands for the path to the directory where you placed configuration files
	 * such as the "runtimelogger-config.xml" file), then your
	 * path_to_your_config_dir is placed first in the list (in case the
	 * directory exists).See {@link RuntimeConsts#SYS_PROP_CONFIG_DIR}
	 * 
	 * @return A list containing potential application configuration folders.
	 * 
	 * @throws IOException Thrown in case there were {@link File} related
	 *         problems determining the folders.
	 */
	public static List<File> toAppConfigDirs() throws IOException {
		List<File> theDirs = new ArrayList<File>();
		String theConfigPath = System.getProperty( RuntimeConsts.SYS_PROP_CONFIG_DIR );
		if ( theConfigPath != null ) {
			File theConfigFile = new File( theConfigPath );
			if ( theConfigFile.exists() && theConfigFile.isDirectory() ) {
				theDirs.add( theConfigFile );
			}
		}
		File theAppBaseDir = toAppBaseDir();
		File eConfigDir;
		if ( theAppBaseDir != null && theAppBaseDir.exists() && theAppBaseDir.isDirectory() && theAppBaseDir.canRead() ) {
			// Determine according child folders:
			for ( String eDirName : RuntimeConsts.CONFIG_DIR_NAMES ) {
				eConfigDir = new File( theAppBaseDir, eDirName );
				if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
					theDirs.add( eConfigDir );
				}
			}
			// Add this folder:
			theDirs.add( theAppBaseDir );
			// Determine according parent folders:
			for ( String eDirName : RuntimeConsts.CONFIG_DIR_NAMES ) {
				eConfigDir = new File( theAppBaseDir.getParentFile(), eDirName );
				if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
					theDirs.add( eConfigDir );
				}
			}
		}
		return theDirs;
	}

	/**
	 * Loads a properties file from the first folder containing such a file as
	 * of the specification for the method
	 * {@link RuntimeUtility#toAppConfigDirs()}.
	 * 
	 * @param aPropertiesFileName The filename of the properties file to load.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 */
	public static Properties loadProperties( String aPropertiesFileName ) throws IOException {
		return new RuntimePropertiesImpl( aPropertiesFileName );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines whether the given {@link StackTraceElement} is not relevant
	 * determining you as the caller
	 * 
	 * @param aStackTraceElement The {@link StackTraceElement} to analyze
	 *        whether it is to be skipped or not.
	 * 
	 * @return True in case you better skip this one as it does not seem to be
	 *         you.
	 */
	private static boolean isSkipStackTraceElement( StackTraceElement aStackTraceElement ) {
		return aStackTraceElement.getClassName().equals( Thread.class.getName() ) || aStackTraceElement.getClassName().equals( RuntimeUtility.class.getName() ) || aStackTraceElement.getLineNumber() <= 1;
	}
}