// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.tabular;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Date;

import org.refcodes.time.DateFormat;
import org.refcodes.time.DateFormats;
import org.refcodes.time.DateUtility;

/**
 * {@link DateColumnImpl} implements the {@link Column} with support for the
 * {@link Date} type and various default or custom date formats.
 */
public class DateColumnImpl extends AbstractColumn<Date> implements Cloneable {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private DateTimeFormatter[] _dateFormats;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructor of the {@link DateColumnImpl} with support for the default
	 * {@link SimpleDateFormat} instances as defined in the
	 * {@link DateFormats#DEFAULT_DATE_FORMATS}.
	 * 
	 * @param aKey The key to be used.
	 */
	public DateColumnImpl( String aKey ) {
		this( aKey, DateFormats.DEFAULT_DATE_FORMATS.getDateFormats() );
	}

	/**
	 * Constructor of the {@link DateColumnImpl} with support for the provided
	 * {@link SimpleDateFormat} instances.
	 * 
	 * @param aKey The key to be used.
	 * 
	 * @param aDateFormats The {@link SimpleDateFormat} instances used for
	 *        parsing date strings.
	 */
	public DateColumnImpl( String aKey, DateTimeFormatter[] aDateFormats ) {
		super( aKey, Date.class );
		_dateFormats = new DateTimeFormatter[aDateFormats.length];
		for ( int i = 0; i < aDateFormats.length; i++ ) {
			_dateFormats[i] = aDateFormats[i];
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] toStorageStrings( Date aValue ) {

		return new String[] {
				_dateFormats[0].format( Instant.ofEpochMilli( aValue.getTime() ) )
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Date fromStorageStrings( String[] aStringValues ) throws ParseException {
		if ( aStringValues == null || aStringValues.length == 0 ) {
			return null;
		}
		else if ( aStringValues.length == 1 ) {
			try {
				return DateUtility.toDate( aStringValues[0], _dateFormats );
			}
			catch ( DateTimeException e ) {
				TemporalAccessor theTemporal = DateFormat.NORM_DATE_FORMAT.getFormatter().parse( aStringValues[0] );
				Instant theInstant = Instant.from( theTemporal );
				return new Date( theInstant.toEpochMilli() );
			}
		}
		throw new IllegalArgumentException( "The type <" + getType().getName() + "> is not an array type though the number of elements in the provided string array is <" + aStringValues.length + "> whereas only one element is being expected." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toPrintable( Date aValue ) {
		Instant theInstant = Instant.ofEpochMilli( aValue.getTime() );
		return _dateFormats[0].format( theInstant );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}
}
