// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.tabular;

import java.util.ArrayList;

/**
 * Implementation of the {@link Row} interface being {@link Cloneable}.
 *
 * @param <T> The type managed by the {@link Row}.
 */
public class RowImpl<T> extends ArrayList<T> implements Row<T>, Cloneable {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link RowImpl} instance configured with the provided
	 * elements.
	 * 
	 * @param aValues The elements to be contained in the {@link RowImpl} in the
	 *        order as passed.
	 */
	@SafeVarargs
	public RowImpl( T... aValues ) {
		for ( T eValue : aValues ) {
			add( eValue );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public T get( Header<T> aHeader, String aKey ) throws HeaderMismatchException, ColumnMismatchException {
		int theIndex = aHeader.indexOf( aKey );
		if ( theIndex == -1 ) {
			throw new HeaderMismatchException( aKey, "Unknwon header key \"" + aKey + "\", no such column." );
		}
		if ( theIndex > size() - 1 ) {
			throw new HeaderMismatchException( aKey, "Header out of bounds index at <" + theIndex + "> for key \"" + aKey + "\" for row (size <" + size() + ">)." );
		}

		T theValue = get( theIndex );
		Column<? extends T> theColumn = aHeader.get( aKey );
		if ( !theColumn.getType().isAssignableFrom( theValue.getClass() ) ) {
			throw new ColumnMismatchException( theColumn, theValue, "Column type <" + theColumn.getType().getName() + "> mismatch for key \"" + theColumn.getKey() + "\" with value <" + theValue.getClass().getName() + "> at index <" + theIndex + ">." );
		}
		return theValue;
	}
}
