// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.tabular;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.refcodes.data.Delimiter;
import org.refcodes.time.DateFormats;

/**
 * Utility for listing specific functionality.
 */
public final class TabularUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private TabularUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a header for a given string array with the column names to be
	 * used for the header. The data type of the header columns will be
	 * {@link Object}.
	 *
	 * @param <T> the generic type
	 * @param aColumnKeys The column names to be use for the header.
	 * @param aColumnFactory creates column instances depending on the key
	 *        passed. This is useful when special header keys are to have a
	 *        dedicated type such as {@link java.util.Date}. If null is passed
	 *        then just {@link ColumnImpl} instances for type Object.class are
	 *        created.
	 * @return A header constructed from the given column names.
	 */
	public static <T> Header<T> toHeader( String[] aColumnKeys, ColumnFactory<T> aColumnFactory ) {
		return toHeader( Arrays.asList( aColumnKeys ), aColumnFactory );
	}

	/**
	 * Creates a header for a given list of strings containing the column names
	 * to be used for the header. The data type of the header columns will be
	 * {@link Object}.
	 *
	 * @param <T> the generic type
	 * @param aColumnKeys The column names to be use for the header.
	 * @param aColumnFactory creates column instances depending on the key
	 *        passed. This is useful when special header keys are to have a
	 *        dedicated type such as {@link java.util.Date}. If null is passed
	 *        then just {@link ColumnImpl} instances for type {@link Object} are
	 *        created.
	 * @return A header constructed from the given column names.
	 */
	@SuppressWarnings("unchecked")
	public static <T> Header<T> toHeader( Collection<String> aColumnKeys, ColumnFactory<T> aColumnFactory ) {
		Column<? extends T>[] theColumns = new Column[aColumnKeys.size()];
		Iterator<String> e = aColumnKeys.iterator();
		int i = 0;
		while ( e.hasNext() ) {
			theColumns[i] = aColumnFactory.createInstance( e.next() );
			i++;
		}
		return new HeaderImpl<T>( theColumns );
	}

	// /////////////////////////////////////////////////////////////////////////
	// MISCELANEOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The given {@link Record} is taken and only all non null values are taken
	 * for the returned {@link Record}. In case of string objects, only
	 * {@link String} instances with a length greater than zero are taken into
	 * account for the returned {@link Record}.
	 * 
	 * @param aRecord The {@link Record} as input.
	 * 
	 * @return The output {@link Record} without any keys which's values were
	 *         null or in case of strings which's length was zero.
	 */
	public static Record<?> toPurged( Record<?> aRecord ) {
		Record<Object> theRecord = new RecordImpl<Object>();
		Object eValue;
		for ( String eKey : aRecord.keySet() ) {
			eValue = aRecord.get( eKey );
			if ( eValue != null ) {
				if ( eValue instanceof String ) {
					if ( ((String) eValue).length() > 0 ) {
						theRecord.put( eKey, eValue );
					}
				}
				else {
					theRecord.put( eKey, eValue );
				}
			}
		}
		return theRecord;
	}

	// /////////////////////////////////////////////////////////////////////////
	// SET THEORY:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tests whether the given potential subset of {@link Header} is matching
	 * the {@link Record}'s elements in terms of matching the same relevant
	 * attributes of the {@link Column} instances with the elements in the
	 * {@link Record}. I.e. all {@link Column} instances in the subset must
	 * match the elements, even if this has more elements.
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Record} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 *
	 * @param aRecord The {@link Record} to be used for testing.
	 * @param aSupersetOf the superset of
	 * @return true in case all columns in the subset match this' elements or
	 *         the argument passed was null.
	 */
	public static boolean isSubsetOf( Record<?> aRecord, Header<?> aSupersetOf ) {
		if ( aSupersetOf != null ) {
			if ( aSupersetOf.size() < aRecord.size() ) {
				return false;
			}
			Column<?> eColumn = null;
			Object eValue = null;
			for ( String eKey : aRecord.keySet() ) {
				if ( !aSupersetOf.containsKey( eKey ) ) {
					return false;
				}
				eColumn = aSupersetOf.get( eKey );
				eValue = aRecord.get( eKey );
				if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Tests whether the given potential equal set of {@link Header} is matching
	 * the {@link Record}'s elements in terms of matching the same relevant
	 * attributes of the columns with the elements in the {@link Record}. I.e.
	 * all {@link Header} in the subset must match the elements, and the number
	 * of {@link Header} must be the same as the number of elements in the
	 * {@link Record}.
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Record} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 * 
	 * @param aRecord The {@link Record} to be used for testing.
	 * 
	 * @param aEquivalsWith The columns which all must be matched with elements
	 *        of this instance.
	 * 
	 * @return True in case all columns in the subset match this' elements and
	 *         the number of elements matches the number of columns or the
	 *         argument passed was null.
	 */
	public static boolean isEqualWith( Record<?> aRecord, Header<?> aEquivalsWith ) {
		if ( aEquivalsWith != null ) {
			if ( aEquivalsWith.size() != aRecord.size() ) {
				return false;
			}
			Column<?> eColumn = null;
			Object eValue = null;
			for ( String eKey : aRecord.keySet() ) {
				if ( !aEquivalsWith.containsKey( eKey ) ) {
					return false;
				}
				eColumn = aEquivalsWith.get( eKey );
				eValue = aRecord.get( eKey );
				if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Tests whether the {@link Record}'s elements is matching the given
	 * potential superset of {@link Header} in terms of matching the same
	 * relevant attributes of the columns with the elements in the
	 * {@link Record}. I.e. all elements in the {@link Record} must match the
	 * {@link Header} in the superset, even if the superset has more elements.
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Record} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 *
	 * @param aRecord The {@link Record} to be used for testing.
	 * @param aSubsetOf the subset of
	 * @return True in case all elements in this instance match the columns in
	 *         the superset or the argument passed was null.
	 */
	public static boolean isSupersetOf( Record<?> aRecord, Header<?> aSubsetOf ) {
		if ( aSubsetOf != null ) {
			if ( aSubsetOf.size() > aRecord.size() ) {
				return false;
			}
			Object eValue = null;
			Column<?> eColumn = null;
			for ( String eKey : aSubsetOf.keySet() ) {
				if ( !aRecord.containsKey( eKey ) ) {
					return false;
				}
				eColumn = aSubsetOf.get( eKey );
				eValue = aRecord.get( eKey );
				if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Tests whether the given potential {@link Header} subset is matching the
	 * {@link Row}'s elements in terms of matching the same relevant attributes
	 * of the {@link Header} with the elements in the {@link Row}. I.e. all
	 * {@link Column} instances in the subset must match the elements, even if
	 * this has more elements.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Row} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 *
	 * @param aRow The {@link Row} to be used for testing.
	 * @param aSupersetOf the superset of
	 * @return true in case all columns in the subset match this' elements or
	 *         the argument passed was null.
	 */
	public static boolean isSubsetOf( Row<?> aRow, Header<?> aSupersetOf ) {
		if ( aSupersetOf != null ) {
			if ( aSupersetOf.size() < aRow.size() ) {
				return false;
			}
			int theIndex = 0;
			Column<?> eColumn = null;
			for ( Object eValue : aRow ) {
				eColumn = aSupersetOf.get( theIndex );
				if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
					return false;
				}
				theIndex++;
			}
		}
		return true;
	}

	/**
	 * Tests whether the given potential {@link Header} equal set is matching
	 * the {@link Row}'s elements in terms of matching the same relevant
	 * attributes of the {@link Header} with the elements in the {@link Row}.
	 * I.e. all {@link Column} instances in the subset must match the elements,
	 * and the number of columns must be the same as the number of elements in
	 * the {@link Row}.
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Row} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 * 
	 * @param aRow The {@link Row} to be used for testing.
	 * 
	 * @param aEquivalsWith The columns which all must be matched with elements
	 *        of this instance.
	 * 
	 * @return True in case all columns in the subset match this' elements and
	 *         the number of elements matches the number of columns or the
	 *         argument passed was null.
	 */
	public static boolean isEqualWith( Row<?> aRow, Header<?> aEquivalsWith ) {
		if ( aEquivalsWith.size() != aRow.size() ) {
			return false;
		}
		int theIndex = 0;
		Column<?> eColumn;
		for ( Object eValue : aRow ) {
			eColumn = aEquivalsWith.get( theIndex );
			if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
				return false;
			}
			theIndex++;
		}
		return true;
	}

	/**
	 * Tests whether the {@link Row}'s elements is matching the given potential
	 * {@link Header} superset in terms of matching the same relevant attributes
	 * of the {@link Header} with the elements in the {@link Row}. I.e. all
	 * elements in the {@link Row} must match the {@link Column} instances in
	 * the superset, even if the superset has more elements.
	 * 
	 * The test also fails (returns false) in case the types of the instances in
	 * the {@link Row} do not match the according key's type in the
	 * {@link Header} {@link Column} instance or vice versa.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 *
	 * @param aRow The {@link Row} to be used for testing.
	 * @param aSubsetOf the subset of
	 * @return True in case all elements in this instance match the columns in
	 *         the superset or the argument passed was null.
	 */
	public static boolean isSupersetOf( Row<?> aRow, Header<?> aSubsetOf ) {
		if ( aSubsetOf != null ) {
			if ( aSubsetOf.size() > aRow.size() ) {
				return false;
			}
			int theIndex = 0;
			Object eValue = null;
			for ( Column<?> eColumn : aSubsetOf ) {
				eValue = aRow.get( theIndex );
				if ( !eColumn.getType().isAssignableFrom( eValue.getClass() ) ) {
					return false;
				}
				theIndex++;
			}
		}
		return true;
	}

	/**
	 * Returns a {@link Record} just containing the keys as defined in the
	 * {@link Header} and found in the provided {@link Record}.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 *
	 * @param <T> the generic type
	 * @param aRecord The {@link Record} to use.
	 * @param aHeader The {@link Header} being the reference for the expected
	 *        result.
	 * @return The {@link Record} being a subset of the given {@link Header}.
	 * @throws ColumnMismatchException in case the type of a key defined in a
	 *         {@link Column} does not match the type of the according value in
	 *         the {@link Record}.
	 */
	@SuppressWarnings("unchecked")
	public static <T> Record<T> toIntersection( Record<T> aRecord, Header<?> aHeader ) throws ColumnMismatchException {
		Record<T> theRecord = new RecordImpl<T>();
		T eValue;
		for ( Column<?> eColumn : aHeader ) {
			if ( eColumn.contains( aRecord ) ) {
				eValue = (T) eColumn.get( aRecord );
				theRecord.put( eColumn.getKey(), eValue );
			}
		}
		return theRecord;
	}

	/**
	 * Returns a {@link Record} just containing the keys as defined in the
	 * {@link Header}; keys not found in the provided {@link Record} are
	 * ignored.
	 * 
	 * -------------------------------------------------------------------------
	 * TODO: Think about moving this method to the {@link Header}.
	 * -------------------------------------------------------------------------
	 *
	 * @param <T> the generic type
	 * @param aRecord The {@link Record} to use.
	 * @param aHeader The {@link Header} being the reference for the expected
	 *        result.
	 * @return The {@link Record} being a subset of the given {@link Header}.
	 * @throws ColumnMismatchException in case the type of a key defined in a
	 *         {@link Column} does not match the type of the according value in
	 *         the {@link Record}.
	 */
	@SuppressWarnings("unchecked")
	public static <T> Record<T> toSubset( Record<T> aRecord, Header<?> aHeader ) throws ColumnMismatchException {
		Record<T> theRecord = new RecordImpl<T>();
		T eValue;
		for ( Column<?> eColumn : aHeader ) {
			eValue = null;
			if ( eColumn.contains( aRecord ) ) {
				eValue = (T) eColumn.get( aRecord );
			}
			theRecord.put( eColumn.getKey(), eValue );
		}
		return theRecord;
	}

	// /////////////////////////////////////////////////////////////////////////
	// STRING:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates the string representation from from the given record. The key /
	 * value pairs of each item in the record are separated from each other with
	 * the assignment operator "=" and the items are separated from each other
	 * by the default CSV separator character and date objects are formatted
	 * with the default date format.
	 * 
	 * @param aRecord The record for which to create a string.
	 * 
	 * @return A string representation of the given record.
	 */
	public static String toString( Record<?> aRecord ) {
		return toString( aRecord, Delimiter.CSV.getChar(), DateFormats.DEFAULT_DATE_FORMATS.getDateFormats() );
	}

	/**
	 * Creates the string representation from from the given record. The key /
	 * value pairs of each item in the record are separated from each other with
	 * the assignment operator "=" and the items are separated from each other
	 * by the given separator character.
	 * 
	 * @param aRecord The record for which to create a string.
	 * @param aSeparator The separator to separate the items (key/value-pairs)
	 *        of the record from each other.
	 * @param aDateFormats The date formats to use when formatting date objects.
	 * 
	 * @return A string representation of the given record.
	 */
	public static String toString( Record<?> aRecord, char aSeparator, DateTimeFormatter[] aDateFormats ) {
		StringBuffer theBuffer = new StringBuffer();
		Object eValue;
		String eKey;
		Iterator<String> e = aRecord.keySet().iterator();
		while ( e.hasNext() ) {
			eKey = e.next();
			theBuffer.append( eKey );
			theBuffer.append( '=' );
			eValue = aRecord.get( eKey );
			if ( eValue instanceof Date ) {
				Instant theInstant = Instant.ofEpochMilli( ((Date) eValue).getTime() );
				theBuffer.append( aDateFormats[0].format( theInstant ) );
			}
			else if ( eValue != null ) {
				theBuffer.append( toString( eValue ) );
			}
			if ( e.hasNext() ) {
				theBuffer.append( aSeparator );
			}
		}
		return theBuffer.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// SEPARATED VALUES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns a separated values representation of the implementing collection
	 * by separating each item with the default separator {@link Delimiter#CSV}.
	 * The common CSV conventions are to be obeyed (although there is none CSV
	 * standard). In case a value's string representation contains the delimiter
	 * char, then this char must be escaped (i.e. by using the backslash '\').
	 * 
	 * @param aRecord The record from which to generate separated values.
	 * 
	 * @return The aDelimiter separated string.
	 */
	public static String toSeparatedValues( Record<?> aRecord ) {
		return toSeparatedValues( aRecord, Delimiter.CSV.getChar() );
	}

	/**
	 * Returns a separated values representation of the implementing collection
	 * by separating each item with the given separator. The common CSV
	 * conventions are to be obeyed (although there is none CSV standard). In
	 * case a value's string representation contains the delimiter char, then
	 * this char must be escaped (i.e. by using the backslash '\').
	 * 
	 * @param aRecord The record from which to generate separated values.
	 * @param aDelimiter The delimiter to use when separating the values.
	 * 
	 * @return The aDelimiter separated string.
	 */
	public static String toSeparatedValues( Record<?> aRecord, char aDelimiter ) {
		StringBuffer theBuffer = new StringBuffer();
		String eKey;
		Object eValue;
		Iterator<String> e = aRecord.keySet().iterator();
		while ( e.hasNext() ) {
			eKey = e.next();
			theBuffer.append( eKey );
			theBuffer.append( '=' );
			eValue = aRecord.get( eKey );
			if ( eValue != null ) {
				theBuffer.append( toString( eValue ) );
			}
			if ( e.hasNext() ) {
				theBuffer.append( aDelimiter );
			}
		}
		return theBuffer.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Converts a value or an array to a string or string array without having a
	 * {@link Column} available.
	 * 
	 * @param aValue The value to convert to a string or a string array.
	 * 
	 * @return A plain string or a string array depending on the type of the
	 *         given value.
	 */
	private static Object toString( Object aValue ) {
		if ( aValue instanceof Object[] ) {
			Object[] theValues = (Object[]) aValue;
			String[] theResult = new String[theValues.length];
			for ( int i = 0; i < theValues.length; i++ ) {
				theResult[i] = theValues[i].toString();
			}
			return theResult;
		}
		else {
			return aValue.toString();
		}
	}
}
