// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.time;

import java.time.ZoneId;
import java.time.format.DateTimeFormatter;

/**
 * The {@link DateFormat} contains <code>Date</code> and <code>DateFormat</code>
 * related values useful for parsing from and formatting to date {@link String}
 * instances.
 */
public enum DateFormat {

	// /////////////////////////////////////////////////////////////////////////
	// DATE FORMATS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Date format as used for cookies as of the Netscape cookie specification.
	 * See
	 * "https://de.wikipedia.org/wiki/HTTP-Cookie#Cookies_nach_Netscape-Spezifikation"
	 */
	NETSCAPE_COOKIE_DATE_FORMAT(DateTimeFormatter.ofPattern( "EEE',' dd-MMM-yyyy HH:mm:ss 'GMT'" ).withZone( ZoneId.of( "GMT" ) )),

	/**
	 * An alternate cookie date format also seen in documentations out there on
	 * the internet.
	 */
	ALTERNATE_COOKIE_DATE_FORMAT(DateTimeFormatter.ofPattern( "EEE',' dd MMM yyyy HH:mm:ss 'GMT'" ).withZone( ZoneId.of( "GMT" ) )),

	/**
	 * Minimum sortable (first comes the year, last the day) date format,
	 * contains the date without the time.
	 */
	MIN_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyy-MM-dd" ).withZone( ZoneId.systemDefault() )),

	/**
	 * Normal sortable (first comes the year, last the seconds) date format,
	 * contains the date and the time till a granularity of seconds. The
	 * character 'T' separates the date portion from the time portion of the
	 * date format in order to unambiguously parse a such formatted date
	 * {@link String}.
	 */
	NORM_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyy-MM-dd'T'HH:mm:ss" ).withZone( ZoneId.systemDefault() )),

	/**
	 * Maximum sortable (first comes the year, last the nanoseconds) date
	 * format, contains the date and the time till a granularity of
	 * milliseconds.The character 'T' separates the date portion from the time
	 * portion of the date format in order to unambiguously parse a such
	 * formatted date {@link String}.
	 */
	MAX_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyy-MM-dd'T'HH:mm:ss.SSSSSSS" ).withZone( ZoneId.systemDefault() )),

	/**
	 * Similar date format as {@link #NORM_DATE_FORMAT} though not avoiding the
	 * space between the date and the time portion of the date format.
	 */
	ALTERNATE_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyy-MM-dd HH:mm:ss" ).withZone( ZoneId.systemDefault() )),

	/**
	 * Date format as used in German speaking locations, first comes the day,
	 * last the year, all separated by a dot '.' character.
	 */
	DE_DATE_FORMAT(DateTimeFormatter.ofPattern( "dd.mm.yyyy" ).withZone( ZoneId.systemDefault() )),

	/**
	 * Date format usable for (parts of) a filename, e.g. for rotating log files
	 * or the such with a granularity of days.
	 */
	MIN_FILENAME_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyyMMdd" ).withZone( ZoneId.systemDefault() )),
	/**
	 * Date format usable for (parts of) a filename, e.g. for rotating log files
	 * or the such with a granularity of seconds.
	 */
	NORM_FILENAME_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyyMMdd'T'HHmmss" ).withZone( ZoneId.systemDefault() )),
	/**
	 * Date format usable for (parts of) a filename, e.g. for rotating log files
	 * or the such with a granularity of nanoseconds.
	 */
	MAX_FILENAME_DATE_FORMAT(DateTimeFormatter.ofPattern( "yyyyMMdd'T'HHmmssSSSSSSS" ).withZone( ZoneId.systemDefault() ));

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// ////////////////////////////////////////////////////////////////////////

	private DateTimeFormatter _dateFormat;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// ////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new date format.
	 *
	 * @param aDateFormat the date format
	 */
	private DateFormat( DateTimeFormatter aDateFormat ) {
		_dateFormat = aDateFormat;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// ////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the formatter.
	 *
	 * @return the formatter
	 */
	public DateTimeFormatter getFormatter() {
		return _dateFormat;
	}
}
