/*
 * Scheduler.java February 2008
 *
 * Copyright (C) 2008, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import static java.nio.channels.SelectionKey.OP_WRITE;

import java.io.IOException;

import org.simpleframework.transport.reactor.Operation;
import org.simpleframework.transport.reactor.Reactor;

/**
 * The <code>Scheduler</code> object is used to schedule a task for
 * execution when it is write ready. This is used by the socket
 * flusher to ensure that the writing thread can be blocked until
 * such time as all the bytes required to be written are written.
 * 
 * @author Niall Gallagher
 *
 * @see org.simpleframework.transport.SocketFlusher
 */
class Scheduler {
   
   /**
    * This is the operation that is scheduled for execution.
    */
   private Operation task;
   
   /**
    * This is the reactor to used to execute the operation.
    */
   private Reactor reactor;
   
   /**
    * This is the lock that is used to signal a blocked thread.
    */
   private Object lock;
   
   /**
    * This is used to determine if the scheduler is running.
    */
   private volatile boolean running;
   
   /**
    * Constructor for the <code>Scheduler</code> object. This is
    * used to create a scheduler that will execute the provided
    * task when the associated socket is write ready. 
    * 
    * @param reactor this is the rector used to schedule execution
    * @param task this is the task that is executed when writable
    * @param lock this is the lock used to signal blocking threads
    */
   public Scheduler(Reactor reactor, Operation task, Object lock) {
      this.reactor = reactor;
      this.task = task;
      this.lock = lock;
   }
   
   /**
    * This is used to repeat schedule the operation for execution.
    * This is executed if the operation has not fully completed
    * its task. If the scheduler is not in a running state then
    * this will not schedule the task for a repeat execution.
    */
   public void repeat() throws IOException {
      if(running) {
         reactor.process(task, OP_WRITE);
      }
   }

   /**
    * This is used to schedule the task for execution. If this is
    * given a boolean true to indicate that it wishes to block
    * then this will block the calling thread until such time as
    * the <code>ready</code> method is invoked.
    * 
    * @param block indicates whether the thread should block
    */
   public void schedule(boolean block) throws IOException {
      if(!running) {
         reactor.process(task, OP_WRITE);
         running = true;
      }
      if(block) {
         listen();
      }
   }
   
   /**
    * This is used to listen for a notification from the reactor to
    * tell the thread that the write operation has completed. If
    * the thread is interrupted upon this call then this will throw
    * an <code>IOException</code> with the root cause.
    */
   private void listen() throws IOException {
      try {
         lock.wait();
      } catch(Exception e) {
         throw new TransportException("Schedule error", e);
      }
   }
   
   /**
    * This is used to notify any waiting threads that they no longer
    * need to wait. This is used when the flusher no longer needs
    * the waiting thread to block. Such an occurence happens when
    * all shared data has been written or has been duplicated.    
    */
   public void release() {
      lock.notifyAll();
   }
   
   /**
    * This is used to signal any blocking threads to wake up. When
    * this is invoked blocking threads are signalled and they can
    * return. This is typically done when the task has finished.
    */
   public void ready() {
      lock.notifyAll();
      running = false;
   }
}
