/*
 * Contract.java May 2004
 *
 * Copyright (C) 2004, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.lease;

import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

/**
 * A <code>Contract</code> is used to represent the contract a
 * lease has been issued. This contains all relevant information
 * regarding the lease, such as the keyed resource that has been 
 * leased and the duration of the lease. Delays for the contract
 * can be measured in any <code>TimeUnit</code> for convinienct.
 *
 * @author Niall Gallagher
 */
interface Contract<T> extends Delayed {
   
   /**
    * This returns the key for the resource this represents. 
    * This is used when the contract has expired to clean resources
    * associated with the lease. It is passed in to the cleaner as
    * an parameter to the callback. The cleaner is then responsible
    * for cleaning any resources associated with the lease.    
    *
    * @return returns the resource key that this represents
    */
   public T getKey();
   
   /**
    * This method will return the number of <code>TimeUnit</code>
    * seconds that remain in the contract. If the value returned is 
    * less than or equal to zero then it should be assumed that the 
    * lease has expired, if greater than zero the lease is active.
    *
    * @return returns the duration in time unit remaining
    */
   public long getDelay(TimeUnit unit);  
   
   /**
    * This method is used to set the number of <code>TimeUnit</code>
    * seconds that should remain within the contract. This is used
    * when the contract is to be reissued. Once a new duration has
    * been set the contract for the lease has been changed and the
    * previous expiry time is ignores, so only one clean is called.
    * 
    * @param delay this is the delay to be used for this contract
    * @param unit this is the time unit measurment for the delay
    */
   public void setDelay(long delay, TimeUnit unit);  
   
   /**
    * This is used to provide a description of the contract that the
    * instance represents. A description well contain the key owned
    * by the contract as well as the expiry time expected for it.
    * This is used to provide descriptive messages in the exceptions.
    * 
    * @return a descriptive message describing the contract object
    */
   public String toString();
}
