package ru.vyarus.gradle.plugin.pom

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.XmlProvider
import org.gradle.api.publish.maven.MavenPom

/**
 * Pom modifications configuration.
 * <p>
 * Example usage:
 * <pre>
 * <code>
 *     maven {
 *        // static model - same as pom in publication (exactly the same!)
 *        pom {
 *             licenses {
 *                  license {
 *                      name = "The MIT License"
 *                  }
 *             }
 *             developers {
 *                 developer {
 *                     id = "dev1"
 *                     name = "Dev1 Name"
 *                     email = "dev1@email.com"
 *                 }
 *             }
 *         }
 *         // raw closure (no structure limitations) - old configuration way
 *         withPom {
 *             licenses {
 *                  license {
 *                      name "The MIT License"
 *                  }
 *             }
 *             developers {
 *                 developer {
 *                     id "dev1"
 *                     name "Dev1 Name"
 *                     email "dev1@email.com"
 *                 }
 *             }
 *         }
 *         // exactly the same as publication's pom.withXml
 *         withPomXml {
 *             asNode().appendNode('tata', 'blabla')
 *         }
 *     }
 * </code>
 * <pre>
 * Multiple declarations could be used.
 * <p>
 * If manual pom modification is required use:
 * <pre><code>
 *     maven.withPomXml {
 *         it.appendNode('description', 'A demonstration of maven POM customization')
 *     }
 * </code></pre>
 * withPomXml usage is equivalent to maven-publish plugin withXml closure, but without need to call
 * asNode() because node is already provided as parameter.
 *
 * @author Vyacheslav Rusakov
 * @since 27.05.2021
 */
@CompileStatic
@SuppressWarnings('ConfusingMethodName')
class PomExtension {

    boolean forcedVersions
    boolean removedDependencyManagement
    boolean disabledScopesCorrection
    boolean disabledBomsReorder
    boolean debug

    protected List<Action<? super MavenPom>> configs = []
    protected List<Closure> rawConfigs = []
    protected List<Action<? super XmlProvider>> xmlModifiers = []
    protected boolean configsApplied

    /**
     * Force dependency versions in the generated pom. Useful when gradle platforms used (by default generated pom
     * would rely on dependenciesManagement section).
     * <p>
     * Activated automatically for {@link #removeDependencyManagement()}.
     * <p>
     * Uses <a
     * href="https://docs.gradle.org/current/userguide/publishing_maven.html#publishing_maven:resolved_dependencies">
     * officially recommended way</a>.
     */
    void forceVersions() {
        forcedVersions = true
    }

    /**
     * Removes dependencyManagement section (generated by gradle for imported platforms). Also, forces dependency
     * versions ({@link #forceVersions()}) because without boms pom become invalid otherwise.
     * <p>
     * Option assumed to be used when gradle platforms used for dependency management, but it is not desired to
     * preserve platforms in the generated pom.
     * <p>
     * Compatible with spring dependency management plugin.
     */
    void removeDependencyManagement() {
        removedDependencyManagement = true
        forceVersions()
    }

    /**
     * Disables dependencies scopes fixes performed by the plugin. May be used to see the default gradle behaviour.
     * <p>
     * Optional and provided configurations dependencies would be generated with correct scopes in any case.
     */
    void disableScopesCorrection() {
        disabledScopesCorrection = true
    }

    /**
     * Gradle java-platform plugin always put used BOMs after other dependencies in the generated pom. Pom plugin fixes
     * this by putting all BOMs at the top (unifies with spring plugin behaviour).
     * <p>
     * Option disables BOMs reordering to see native gradle behaviour.
     */
    void disableBomsReorder() {
        disabledBomsReorder = true
    }

    /**
     * Enable debug logs.
     * Pom model diff is shown only for gradle 8.4 and above.
     */
    void debug() {
        debug = true
    }

    /**
     * Exactly the same as pom configuration in publication, but applied to all publications.
     * <p>
     * ATTENTION: All properties assignment must be done with '='
     *
     * @param config configuration action
     */
    void pom(Action<? super MavenPom> config) {
        if (configsApplied) {
            throw new IllegalStateException('Too late maven.pom() appliance: configuration already applied')
        }
        this.configs.add(config)
    }

    /**
     * Pure groovy closure configuration (old way). Does not limited with any structure and so ANY xml tags could
     * be specified like this.
     * <p>
     * ATTENTION: All properties assignment must be done as "method" - does not contain '='!
     *
     * @param config user pom
     */
    void withPom(Closure config) {
        if (configsApplied) {
            throw new IllegalStateException('Too late maven.withPom() appliance: configuration already applied')
        }
        this.rawConfigs.add(config)
    }

    /**
     * Raw xml modification action. This is the same as "pom.withXml" applied in publication, but this block would
     * be applied in all registered publications.
     *
     * @param modifier manual pom modification closure
     */
    void withPomXml(Action<? super XmlProvider> modifier) {
        if (configsApplied) {
            throw new IllegalStateException('Too late maven.withPomXml() appliance: configuration already applied')
        }
        this.xmlModifiers.add(modifier)
    }

    protected PomExtension copy() {
        // no more changes allowed (they could not be applied)
        configsApplied = true
        return new PomExtension(
                forcedVersions: forcedVersions,
                removedDependencyManagement: removedDependencyManagement,
                disabledScopesCorrection: disabledScopesCorrection,
                disabledBomsReorder: disabledBomsReorder,
                debug: debug,

                configs: configs,
                rawConfigs: rawConfigs,
                xmlModifiers: xmlModifiers
        )
    }
}
