/*
 * Decompiled with CFR 0.152.
 */
package org.opends.server.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import org.opends.server.types.OperatingSystem;
import org.opends.server.types.PublicAPI;
import org.opends.server.types.StabilityLevel;
import org.opends.server.util.SetupUtils;

@PublicAPI(stability=StabilityLevel.VOLATILE, mayInstantiate=true, mayExtend=false, mayInvoke=true)
public final class CertificateManager {
    public static final String KEYTOOL_COMMAND;
    public static final String KEY_STORE_TYPE_JKS = "JKS";
    public static final String KEY_STORE_TYPE_PKCS11 = "PKCS11";
    public static final String KEY_STORE_TYPE_PKCS12 = "PKCS12";
    public static final String KEY_STORE_PATH_PKCS11 = "NONE";
    private KeyStore keyStore;
    private String keyStorePIN;
    private String keyStorePath;
    private String keyStoreType;

    public static boolean mayUseCertificateManager() {
        return KEYTOOL_COMMAND != null;
    }

    public CertificateManager(String keyStorePath, String keyStoreType, String keyStorePIN) throws IllegalArgumentException, NullPointerException, UnsupportedOperationException {
        if (keyStorePath == null || keyStorePath.length() == 0) {
            throw new NullPointerException("keyStorePath");
        }
        if (keyStoreType == null || keyStoreType.length() == 0) {
            throw new NullPointerException("keyStoreType");
        }
        if (keyStorePIN == null || keyStorePIN.length() == 0) {
            throw new NullPointerException("keyStorePIN");
        }
        if (keyStoreType.equals(KEY_STORE_TYPE_PKCS11)) {
            if (!keyStorePath.equals(KEY_STORE_PATH_PKCS11)) {
                throw new IllegalArgumentException("Invalid key store path for PKCS11 keystore -- it must be NONE");
            }
        } else if (keyStoreType.equals(KEY_STORE_TYPE_JKS) || keyStoreType.equals(KEY_STORE_TYPE_PKCS12)) {
            File keyStoreFile = new File(keyStorePath);
            if (keyStoreFile.exists()) {
                if (!keyStoreFile.isFile()) {
                    throw new IllegalArgumentException("Key store path " + keyStorePath + " exists but is not a file.");
                }
            } else {
                File keyStoreDirectory = keyStoreFile.getParentFile();
                if (keyStoreDirectory == null || !keyStoreDirectory.exists() || !keyStoreDirectory.isDirectory()) {
                    throw new IllegalArgumentException("Parent directory for key store path " + keyStorePath + " does not exist or " + "is not a directory.");
                }
            }
        } else {
            throw new IllegalArgumentException("Invalid key store type -- it must be one of JKS, PKCS11, or PKCS12");
        }
        this.keyStorePath = keyStorePath;
        this.keyStoreType = keyStoreType;
        this.keyStorePIN = keyStorePIN;
        this.keyStore = null;
    }

    public boolean aliasInUse(String alias) throws KeyStoreException, NullPointerException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        KeyStore keyStore = this.getKeyStore();
        if (keyStore == null) {
            return false;
        }
        return keyStore.containsAlias(alias);
    }

    public String[] getCertificateAliases() throws KeyStoreException {
        KeyStore keyStore = this.getKeyStore();
        if (keyStore == null) {
            return null;
        }
        Enumeration<String> aliasEnumeration = keyStore.aliases();
        if (aliasEnumeration == null) {
            return new String[0];
        }
        ArrayList<String> aliasList = new ArrayList<String>();
        while (aliasEnumeration.hasMoreElements()) {
            aliasList.add(aliasEnumeration.nextElement());
        }
        String[] aliases = new String[aliasList.size()];
        return aliasList.toArray(aliases);
    }

    public Certificate getCertificate(String alias) throws KeyStoreException, NullPointerException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        KeyStore keyStore = this.getKeyStore();
        if (keyStore == null) {
            throw new KeyStoreException("The key store does not exist.");
        }
        return keyStore.getCertificate(alias);
    }

    public void generateSelfSignedCertificate(String alias, String subjectDN, int validity) throws KeyStoreException, IllegalArgumentException, NullPointerException, UnsupportedOperationException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        if (subjectDN == null || subjectDN.length() == 0) {
            throw new NullPointerException("subjectDN");
        }
        if (validity <= 0) {
            throw new IllegalArgumentException("The validity must be positive.");
        }
        if (KEYTOOL_COMMAND == null) {
            throw new UnsupportedOperationException("The certificate manager may not be used to alter the contents of key stores on this system.");
        }
        if (this.aliasInUse(alias)) {
            throw new IllegalArgumentException("A certificate with alias " + alias + " already exists in the key store.");
        }
        this.keyStore = null;
        String[] commandElements = new String[]{KEYTOOL_COMMAND, this.getGenKeyCommand(), "-alias", alias, "-dname", subjectDN, "-keyalg", "rsa", "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
        commandElements = new String[]{KEYTOOL_COMMAND, "-selfcert", "-alias", alias, "-validity", String.valueOf(validity), "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
    }

    public File generateCertificateSigningRequest(String alias, String subjectDN) throws KeyStoreException, IOException, NullPointerException, UnsupportedOperationException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        if (subjectDN == null || subjectDN.length() == 0) {
            throw new NullPointerException("subjectDN");
        }
        if (KEYTOOL_COMMAND == null) {
            throw new UnsupportedOperationException("The certificate manager may not be used to alter the contents of key stores on this system.");
        }
        if (this.aliasInUse(alias)) {
            throw new IllegalArgumentException("A certificate with alias " + alias + " already exists in the key store.");
        }
        this.keyStore = null;
        String[] commandElements = new String[]{KEYTOOL_COMMAND, this.getGenKeyCommand(), "-alias", alias, "-dname", subjectDN, "-keyalg", "rsa", "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
        File csrFile = File.createTempFile("CertificateManager-", ".csr");
        csrFile.deleteOnExit();
        commandElements = new String[]{KEYTOOL_COMMAND, "-certreq", "-alias", alias, "-file", csrFile.getAbsolutePath(), "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
        return csrFile;
    }

    public void addCertificate(String alias, File certificateFile) throws IllegalArgumentException, KeyStoreException, NullPointerException, UnsupportedOperationException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        if (certificateFile == null) {
            throw new NullPointerException("certificateFile");
        }
        if (!certificateFile.exists() || !certificateFile.isFile()) {
            throw new IllegalArgumentException("Certificate file " + certificateFile.getAbsolutePath() + " does not exist or is not a file.");
        }
        if (KEYTOOL_COMMAND == null) {
            throw new UnsupportedOperationException("The certificate manager may not be used to alter the contents of key stores on this system.");
        }
        this.keyStore = null;
        String[] commandElements = new String[]{KEYTOOL_COMMAND, "-import", "-noprompt", "-alias", alias, "-file", certificateFile.getAbsolutePath(), "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
    }

    public void removeCertificate(String alias) throws IllegalArgumentException, KeyStoreException, NullPointerException, UnsupportedOperationException {
        if (alias == null || alias.length() == 0) {
            throw new NullPointerException("alias");
        }
        if (KEYTOOL_COMMAND == null) {
            throw new UnsupportedOperationException("The certificate manager may not be used to alter the contents of key stores on this system.");
        }
        if (!this.aliasInUse(alias)) {
            throw new IllegalArgumentException("There is no certificate with alias " + alias + " in the key store.");
        }
        this.keyStore = null;
        String[] commandElements = new String[]{KEYTOOL_COMMAND, "-delete", "-alias", alias, "-keystore", this.keyStorePath, "-storetype", this.keyStoreType};
        this.runKeyTool(commandElements, this.keyStorePIN, this.keyStorePIN, true);
    }

    private void runKeyTool(String[] commandElements, String keyStorePassword, String storePassword, boolean outputAcceptable) throws KeyStoreException {
        File keyStoreFile;
        String lineSeparator = System.getProperty("line.separator");
        if (lineSeparator == null) {
            lineSeparator = "\n";
        }
        boolean keyStoreDefined = (keyStoreFile = new File(this.keyStorePath)).exists() && keyStoreFile.length() > 0L || KEY_STORE_TYPE_PKCS11.equals(this.keyStoreType);
        boolean isNewKeyStorePassword = !keyStoreDefined && (this.getGenKeyCommand().equalsIgnoreCase(commandElements[1]) || "-import".equalsIgnoreCase(commandElements[1]));
        boolean isNewStorePassword = this.getGenKeyCommand().equalsIgnoreCase(commandElements[1]);
        boolean askForStorePassword = !"-import".equalsIgnoreCase(commandElements[1]);
        try {
            int bytesRead;
            ProcessBuilder processBuilder = new ProcessBuilder(commandElements);
            processBuilder.redirectErrorStream(true);
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            Process process = processBuilder.start();
            InputStream inputStream = process.getInputStream();
            OutputStream out = process.getOutputStream();
            if (!this.isJDK15() && SetupUtils.getOperatingSystem() == OperatingSystem.AIX) {
                try {
                    Thread.sleep(1500L);
                }
                catch (Throwable t) {
                    // empty catch block
                }
            }
            out.write(keyStorePassword.getBytes());
            out.write(lineSeparator.getBytes());
            out.flush();
            if (!this.isJDK15() && isNewKeyStorePassword) {
                if (SetupUtils.getOperatingSystem() == OperatingSystem.AIX) {
                    try {
                        Thread.sleep(1500L);
                    }
                    catch (Throwable t) {
                        // empty catch block
                    }
                }
                out.write(keyStorePassword.getBytes());
                out.write(lineSeparator.getBytes());
                out.flush();
            }
            if (askForStorePassword) {
                out.write(storePassword.getBytes());
                out.write(lineSeparator.getBytes());
                out.flush();
                if (!this.isJDK15() && isNewStorePassword && SetupUtils.getOperatingSystem() != OperatingSystem.AIX) {
                    out.write(storePassword.getBytes());
                    out.write(lineSeparator.getBytes());
                    out.flush();
                }
            }
            out.close();
            while ((bytesRead = inputStream.read(buffer)) >= 0) {
                if (bytesRead <= 0) continue;
                output.write(buffer, 0, bytesRead);
            }
            process.waitFor();
            int exitValue = process.exitValue();
            byte[] outputBytes = output.toByteArray();
            if (exitValue != 0) {
                StringBuilder message = new StringBuilder();
                message.append("Unexpected exit code of ");
                message.append(exitValue);
                message.append(" returned from the keytool utility.");
                if (outputBytes != null && outputBytes.length > 0) {
                    message.append("  The generated output was:  '");
                    message.append(new String(outputBytes));
                    message.append("'.");
                }
                throw new KeyStoreException(message.toString());
            }
            if (!outputAcceptable && outputBytes != null && outputBytes.length > 0) {
                StringBuilder message = new StringBuilder();
                message.append("Unexpected output generated by the keytool utility:  '");
                message.append(new String(outputBytes));
                message.append("'.");
                throw new KeyStoreException(message.toString());
            }
        }
        catch (KeyStoreException kse) {
            throw kse;
        }
        catch (Exception e) {
            throw new KeyStoreException("Could not invoke the KeyTool.run method:  " + e, e);
        }
    }

    private KeyStore getKeyStore() throws KeyStoreException {
        if (this.keyStore != null) {
            return this.keyStore;
        }
        FileInputStream keyStoreInputStream = null;
        if (this.keyStoreType.equals(KEY_STORE_TYPE_JKS) || this.keyStoreType.equals(KEY_STORE_TYPE_PKCS12)) {
            File keyStoreFile = new File(this.keyStorePath);
            if (!keyStoreFile.exists()) {
                return null;
            }
            try {
                keyStoreInputStream = new FileInputStream(keyStoreFile);
            }
            catch (Exception e) {
                throw new KeyStoreException(String.valueOf(e), e);
            }
        }
        KeyStore keyStore = KeyStore.getInstance(this.keyStoreType);
        try {
            keyStore.load(keyStoreInputStream, this.keyStorePIN.toCharArray());
            KeyStore e = this.keyStore = keyStore;
            return e;
        }
        catch (Exception e) {
            throw new KeyStoreException(String.valueOf(e), e);
        }
        finally {
            if (keyStoreInputStream != null) {
                try {
                    keyStoreInputStream.close();
                }
                catch (Throwable t) {}
            }
        }
    }

    private boolean isJDK15() {
        boolean isJDK15 = false;
        try {
            String javaRelease = System.getProperty("java.version");
            isJDK15 = javaRelease.startsWith("1.5");
        }
        catch (Throwable t) {
            System.err.println("Cannot get the java version: " + t);
        }
        return isJDK15;
    }

    private String getGenKeyCommand() {
        String genKeyCommand = !this.isJDK15() ? "-genkeypair" : "-genkey";
        return genKeyCommand;
    }

    static {
        String keytoolCommand = null;
        try {
            String cmd = System.getProperty("java.home") + File.separator + "bin" + File.separator + "keytool";
            File cmdFile = new File(cmd);
            keytoolCommand = cmdFile.exists() ? cmdFile.getAbsolutePath() : ((cmdFile = new File(cmd = cmd + ".exe")).exists() ? cmdFile.getAbsolutePath() : null);
        }
        catch (Exception e) {
            keytoolCommand = null;
        }
        KEYTOOL_COMMAND = SetupUtils.getScriptPath(keytoolCommand);
    }
}

