/*
 * Decompiled with CFR 0.152.
 */
package com.android.utils;

import com.android.utils.PathUtils;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.FileWriteMode;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import kotlin.io.FilesKt;
import org.jetbrains.annotations.NotNull;

public final class FileUtils {
    private static final Joiner PATH_JOINER = Joiner.on((char)File.separatorChar);
    private static final Joiner COMMA_SEPARATED_JOINER = Joiner.on((String)", ");
    private static final Joiner UNIX_NEW_LINE_JOINER = Joiner.on((char)'\n');

    private FileUtils() {
    }

    public static void deletePath(File path) throws IOException {
        FileUtils.deleteRecursivelyIfExists(path);
    }

    public static void deleteDirectoryContents(File directory) throws IOException {
        Preconditions.checkArgument((boolean)directory.isDirectory(), (Object)"!directory.isDirectory");
        File[] files = directory.listFiles();
        Preconditions.checkNotNull((Object)files);
        for (File file2 : files) {
            FileUtils.deletePath(file2);
        }
    }

    public static void cleanOutputDir(File path) throws IOException {
        if (!path.isDirectory()) {
            if (path.exists()) {
                FileUtils.deletePath(path);
            }
            if (!path.mkdirs()) {
                throw new IOException(String.format("Could not create empty folder %s", path));
            }
            return;
        }
        FileUtils.deleteDirectoryContents(path);
    }

    public static void copyFile(File from, File to) throws IOException {
        FileUtils.copyFile(from.toPath(), to.toPath());
    }

    public static void copyFile(Path from, Path to) throws IOException {
        if (System.getProperty("os.name").toLowerCase(Locale.US).contains("windows")) {
            FileUtils.copyFile(from, to, StandardCopyOption.REPLACE_EXISTING);
        } else {
            FileUtils.copyFile(from, to, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void copyFile(Path from, Path to, CopyOption ... options) throws IOException {
        Files.copy(from, to, options);
        FileUtils.setWritable(to);
    }

    private static void setWritable(@NotNull Path path) {
        File fileOrNull;
        try {
            fileOrNull = path.toFile();
        }
        catch (UnsupportedOperationException e) {
            fileOrNull = null;
        }
        if (fileOrNull != null && !fileOrNull.canWrite()) {
            fileOrNull.setWritable(true);
        }
    }

    public static void copyDirectory(File from, File to) throws IOException {
        FileUtils.copyDirectory(from.toPath(), to.toPath(), true);
    }

    public static void copyDirectory(Path from, Path to, boolean copyDotFiles) throws IOException {
        FileUtils.copyDirectory(from, to, path -> copyDotFiles || !path.getFileName().toString().startsWith("."));
    }

    public static void copyDirectory(final Path from, final Path to, final Predicate<Path> filter) throws IOException {
        Preconditions.checkArgument((boolean)Files.isDirectory(from, new LinkOption[0]), (Object)("\"" + from.toAbsolutePath() + "\" is not a directory."));
        Preconditions.checkArgument((Files.notExists(to, LinkOption.NOFOLLOW_LINKS) || Files.isDirectory(to, LinkOption.NOFOLLOW_LINKS) ? 1 : 0) != 0, (Object)("\"" + from.toAbsolutePath() + "\" exists and is not a directory."));
        Preconditions.checkArgument((!FileUtils.isAncestor(from, to, false) ? 1 : 0) != 0, (Object)("\"" + from.toAbsolutePath() + "\" is ancestor of \"" + to + "\". Can't copy to itself"));
        Files.walkFileTree(from, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @NotNull
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (!dir.equals(from) && !filter.test(dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Files.createDirectories(this.sourceToTarget(dir), new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NotNull
            public FileVisitResult visitFile(Path sourceFile, BasicFileAttributes attrs) throws IOException {
                if (filter.test(sourceFile)) {
                    FileUtils.copyFile(sourceFile, this.sourceToTarget(sourceFile));
                }
                return FileVisitResult.CONTINUE;
            }

            @NotNull
            private Path sourceToTarget(Path sourceFile) {
                return to.resolve(from.relativize(sourceFile).toString());
            }
        });
    }

    static boolean isAncestor(@NotNull Path p1, @NotNull Path p2, boolean strict) {
        Path relative;
        try {
            relative = FileUtils.realPath(p1).relativize(FileUtils.realPath(p2));
        }
        catch (IllegalArgumentException e) {
            return false;
        }
        int segmentCount = relative.getNameCount();
        if (segmentCount == 0) {
            return !strict;
        }
        return !relative.getName(0).toString().equals("..");
    }

    @NotNull
    public static Path realPath(@NotNull Path path) {
        try {
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException ignore) {
            return path;
        }
    }

    public static void copyFileToDirectory(File from, File to) throws IOException {
        FileUtils.copyFile(from, new File(to, from.getName()));
    }

    public static void copyDirectoryToDirectory(File from, File to) throws IOException {
        FileUtils.copyDirectory(from, new File(to, from.getName()));
    }

    public static void copyDirectoryContentToDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.isDirectory(), (Object)"Source path is not a directory.");
        File[] children = from.listFiles();
        if (children != null) {
            for (File f : children) {
                File destination;
                if (f.isDirectory()) {
                    destination = new File(to, FilesKt.toRelativeString((File)f, (File)from));
                    com.google.common.io.Files.createParentDirs((File)destination);
                    FileUtils.mkdirs(destination);
                    FileUtils.copyDirectoryContentToDirectory(f, destination);
                    continue;
                }
                if (!f.isFile()) continue;
                destination = new File(to, FilesKt.toRelativeString((File)f.getParentFile(), (File)from));
                com.google.common.io.Files.createParentDirs((File)destination);
                FileUtils.mkdirs(destination);
                FileUtils.copyFileToDirectory(f, destination);
            }
        }
    }

    public static File mkdirs(File folder) {
        if (!folder.mkdirs() && !folder.isDirectory()) {
            throw new RuntimeException("Cannot create directory " + folder);
        }
        return folder;
    }

    public static void delete(File file2) throws IOException {
        Files.delete(file2.toPath());
    }

    public static void deleteIfExists(File file2) throws IOException {
        Files.deleteIfExists(file2.toPath());
    }

    public static void deleteRecursivelyIfExists(File file2) throws IOException {
        PathUtils.deleteRecursivelyIfExists(file2.toPath());
    }

    public static void renameTo(File file2, File to) throws IOException {
        boolean result = file2.renameTo(to);
        if (!result) {
            throw new IOException("Failed to rename " + file2.getAbsolutePath() + " to " + to);
        }
    }

    public static File join(File dir, String ... paths) {
        if (paths.length == 0) {
            return dir;
        }
        return new File(dir, PATH_JOINER.join((Object[])paths));
    }

    public static File join(File dir, Iterable<String> paths) {
        return new File(dir, PATH_JOINER.join(FileUtils.removeEmpty(paths)));
    }

    public static String join(String ... paths) {
        return PATH_JOINER.join(FileUtils.removeEmpty(Lists.newArrayList((Object[])paths)));
    }

    public static String join(Iterable<String> paths) {
        return PATH_JOINER.join(paths);
    }

    private static Iterable<String> removeEmpty(Iterable<String> input) {
        return Lists.newArrayList(input).stream().filter(it -> !it.isEmpty()).collect(Collectors.toList());
    }

    public static String loadFileWithUnixLineSeparators(File file2) throws IOException {
        return UNIX_NEW_LINE_JOINER.join(Files.readAllLines(file2.toPath()));
    }

    public static String toSystemDependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace('/', File.separatorChar);
        }
        return path;
    }

    public static String escapeSystemDependentCharsIfNecessary(String path) {
        if (File.separatorChar == '\\') {
            return path.replace("\\", "\\\\");
        }
        return path;
    }

    public static String toSystemIndependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        return path;
    }

    public static String toExportableSystemDependentPath(File file2) {
        if (File.separatorChar != '/' && !file2.getAbsolutePath().startsWith("\\\\?\\")) {
            return "\\\\?\\" + file2.getAbsolutePath();
        }
        return file2.getAbsolutePath();
    }

    public static String sha1(File file2) throws IOException {
        return Hashing.sha1().hashBytes(com.google.common.io.Files.toByteArray((File)file2)).toString();
    }

    public static FluentIterable<File> getAllFiles(File dir) {
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)dir)).filter(com.google.common.io.Files.isFile());
    }

    public static String getNamesAsCommaSeparatedList(Iterable<File> files) {
        return COMMA_SEPARATED_JOINER.join(Iterables.transform(files, File::getName));
    }

    public static String sanitizeFileName(String input) {
        return input.replaceAll("[:\\\\/*\"?|<>']", "_");
    }

    public static String getDirectoryNameForJar(File inputFile) {
        HashFunction hashFunction2 = Hashing.sha1();
        HashCode hashCode = hashFunction2.hashString((CharSequence)inputFile.getAbsolutePath(), Charsets.UTF_16LE);
        Object name = com.google.common.io.Files.getNameWithoutExtension((String)inputFile.getName());
        if (((String)name).equals("classes") && inputFile.getAbsolutePath().contains("exploded-aar")) {
            File versionDir = inputFile.getParentFile().getParentFile();
            File artifactDir = versionDir.getParentFile();
            File groupDir = artifactDir.getParentFile();
            name = Joiner.on((char)'-').join((Object)groupDir.getName(), (Object)artifactDir.getName(), new Object[]{versionDir.getName()});
        }
        name = (String)name + "_" + hashCode.toString();
        return name;
    }

    public static void createFile(File file2, String content) throws IOException {
        Preconditions.checkArgument((!file2.exists() ? 1 : 0) != 0, (String)"%s exists already.", (Object)file2);
        FileUtils.writeToFile(file2, content);
    }

    public static void writeToFile(File file2, String content) throws IOException {
        com.google.common.io.Files.createParentDirs((File)file2);
        com.google.common.io.Files.asCharSink((File)file2, (Charset)StandardCharsets.UTF_8, (FileWriteMode[])new FileWriteMode[0]).write((CharSequence)content);
    }

    public static List<File> find(File base, Pattern pattern) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)base)).filter(file2 -> pattern.matcher(FileUtils.toSystemIndependentPath(file2.getPath())).find()).toList();
    }

    public static Optional<File> find(File base, String name) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)base)).filter(file2 -> name.equals(file2.getName())).last();
    }

    public static String joinFilePaths(Iterable<File> files) {
        return Joiner.on((char)File.pathSeparatorChar).join(Iterables.transform(files, File::getAbsolutePath));
    }

    public static boolean parentDirExists(File file2) {
        File canonicalFile;
        try {
            canonicalFile = file2.getCanonicalFile();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return canonicalFile.getParentFile() != null && canonicalFile.getParentFile().exists();
    }

    public static boolean isFileInDirectory(File fileOrDir, File baseDir) {
        String relativePath;
        try {
            relativePath = baseDir.toPath().relativize(fileOrDir.toPath()).toString();
        }
        catch (IllegalArgumentException e) {
            return false;
        }
        return !relativePath.isEmpty() && !relativePath.startsWith("..");
    }

    public static boolean isSameFile(File file1, File file2) {
        try {
            if (file1.exists() && file2.exists()) {
                return Files.isSameFile(file1.toPath(), file2.toPath());
            }
            return file1.getCanonicalFile().equals(file2.getCanonicalFile());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static FileSystem createZipFilesystem(Path archive) throws IOException {
        URI uri = URI.create("jar:" + archive.toUri().toString());
        return FileSystems.newFileSystem(uri, Collections.emptyMap());
    }
}

