/*
 * Decompiled with CFR 0.152.
 */
package com.android.utils;

import com.android.utils.PathUtils;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.FileWriteMode;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import kotlin.io.FilesKt;
import org.jetbrains.annotations.NotNull;

public final class FileUtils {
    private static final Joiner PATH_JOINER = Joiner.on(File.separatorChar);
    private static final Joiner COMMA_SEPARATED_JOINER = Joiner.on(", ");
    private static final Joiner UNIX_NEW_LINE_JOINER = Joiner.on('\n');

    private FileUtils() {
    }

    public static void deletePath(File path) throws IOException {
        FileUtils.deleteRecursivelyIfExists(path);
    }

    public static void deleteDirectoryContents(File directory) throws IOException {
        Preconditions.checkArgument(directory.isDirectory(), "!directory.isDirectory");
        File[] files2 = directory.listFiles();
        Preconditions.checkNotNull(files2);
        for (File file2 : files2) {
            FileUtils.deletePath(file2);
        }
    }

    public static void cleanOutputDir(File path) throws IOException {
        if (!path.isDirectory()) {
            if (path.exists()) {
                FileUtils.deletePath(path);
            }
            if (!path.mkdirs()) {
                throw new IOException(String.format("Could not create empty folder %s", path));
            }
            return;
        }
        FileUtils.deleteDirectoryContents(path);
    }

    public static void copyFile(File from2, File to) throws IOException {
        FileUtils.copyFile(from2.toPath(), to.toPath());
    }

    public static void copyFile(Path from2, Path to) throws IOException {
        if (System.getProperty("os.name").toLowerCase(Locale.US).contains("windows")) {
            FileUtils.copyFile(from2, to, StandardCopyOption.REPLACE_EXISTING);
        } else {
            FileUtils.copyFile(from2, to, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void copyFile(Path from2, Path to, CopyOption ... options) throws IOException {
        Files.copy(from2, to, options);
        FileUtils.setWritable(to);
    }

    private static void setWritable(@NotNull Path path) {
        File fileOrNull2;
        try {
            fileOrNull2 = path.toFile();
        }
        catch (UnsupportedOperationException e2) {
            fileOrNull2 = null;
        }
        if (fileOrNull2 != null && !fileOrNull2.canWrite()) {
            fileOrNull2.setWritable(true);
        }
    }

    public static void copyDirectory(File from2, File to) throws IOException {
        FileUtils.copyDirectory(from2.toPath(), to.toPath(), true);
    }

    public static void copyDirectory(Path from2, Path to, boolean copyDotFiles) throws IOException {
        FileUtils.copyDirectory(from2, to, path -> copyDotFiles || !path.getFileName().toString().startsWith("."));
    }

    public static void copyDirectory(final Path from2, final Path to, final Predicate<Path> filter2) throws IOException {
        Preconditions.checkArgument(Files.isDirectory(from2, new LinkOption[0]), "\"" + from2.toAbsolutePath() + "\" is not a directory.");
        Preconditions.checkArgument(Files.notExists(to, LinkOption.NOFOLLOW_LINKS) || Files.isDirectory(to, LinkOption.NOFOLLOW_LINKS), "\"" + from2.toAbsolutePath() + "\" exists and is not a directory.");
        Preconditions.checkArgument(!FileUtils.isAncestor(from2, to, false), "\"" + from2.toAbsolutePath() + "\" is ancestor of \"" + to + "\". Can't copy to itself");
        Files.walkFileTree(from2, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @NotNull
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (!dir.equals(from2) && !filter2.test(dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Files.createDirectories(this.sourceToTarget(dir), new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NotNull
            public FileVisitResult visitFile(Path sourceFile, BasicFileAttributes attrs) throws IOException {
                if (filter2.test(sourceFile)) {
                    FileUtils.copyFile(sourceFile, this.sourceToTarget(sourceFile));
                }
                return FileVisitResult.CONTINUE;
            }

            @NotNull
            private Path sourceToTarget(Path sourceFile) {
                return to.resolve(from2.relativize(sourceFile).toString());
            }
        });
    }

    static boolean isAncestor(@NotNull Path p1, @NotNull Path p2, boolean strict) {
        Path relative;
        try {
            relative = FileUtils.realPath(p1).relativize(FileUtils.realPath(p2));
        }
        catch (IllegalArgumentException e2) {
            return false;
        }
        int segmentCount = relative.getNameCount();
        if (segmentCount == 0) {
            return !strict;
        }
        return !relative.getName(0).toString().equals("..");
    }

    @NotNull
    public static Path realPath(@NotNull Path path) {
        try {
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException ignore) {
            return path;
        }
    }

    public static void copyFileToDirectory(File from2, File to) throws IOException {
        FileUtils.copyFile(from2, new File(to, from2.getName()));
    }

    public static void copyDirectoryToDirectory(File from2, File to) throws IOException {
        FileUtils.copyDirectory(from2, new File(to, from2.getName()));
    }

    public static void copyDirectoryContentToDirectory(File from2, File to) throws IOException {
        Preconditions.checkArgument(from2.isDirectory(), "Source path is not a directory.");
        File[] children2 = from2.listFiles();
        if (children2 != null) {
            for (File f2 : children2) {
                File destination;
                if (f2.isDirectory()) {
                    destination = new File(to, FilesKt.toRelativeString(f2, from2));
                    com.google.common.io.Files.createParentDirs(destination);
                    FileUtils.mkdirs(destination);
                    FileUtils.copyDirectoryContentToDirectory(f2, destination);
                    continue;
                }
                if (!f2.isFile()) continue;
                destination = new File(to, FilesKt.toRelativeString(f2.getParentFile(), from2));
                com.google.common.io.Files.createParentDirs(destination);
                FileUtils.mkdirs(destination);
                FileUtils.copyFileToDirectory(f2, destination);
            }
        }
    }

    public static File mkdirs(File folder) {
        if (!folder.mkdirs() && !folder.isDirectory()) {
            throw new RuntimeException("Cannot create directory " + folder);
        }
        return folder;
    }

    public static void delete(File file2) throws IOException {
        Files.delete(file2.toPath());
    }

    public static void deleteIfExists(File file2) throws IOException {
        Files.deleteIfExists(file2.toPath());
    }

    public static void deleteRecursivelyIfExists(File file2) throws IOException {
        PathUtils.deleteRecursivelyIfExists(file2.toPath());
    }

    public static void renameTo(File file2, File to) throws IOException {
        boolean result2 = file2.renameTo(to);
        if (!result2) {
            throw new IOException("Failed to rename " + file2.getAbsolutePath() + " to " + to);
        }
    }

    public static File join(File dir, String ... paths2) {
        if (paths2.length == 0) {
            return dir;
        }
        return new File(dir, PATH_JOINER.join(paths2));
    }

    public static File join(File dir, Iterable<String> paths2) {
        return new File(dir, PATH_JOINER.join(FileUtils.removeEmpty(paths2)));
    }

    public static String join(String ... paths2) {
        return PATH_JOINER.join(FileUtils.removeEmpty(Lists.newArrayList(paths2)));
    }

    public static String join(Iterable<String> paths2) {
        return PATH_JOINER.join(paths2);
    }

    private static Iterable<String> removeEmpty(Iterable<String> input2) {
        return Lists.newArrayList(input2).stream().filter(it -> !it.isEmpty()).collect(Collectors.toList());
    }

    public static String loadFileWithUnixLineSeparators(File file2) throws IOException {
        return UNIX_NEW_LINE_JOINER.join(Files.readAllLines(file2.toPath()));
    }

    public static String toSystemDependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace('/', File.separatorChar);
        }
        return path;
    }

    public static String escapeSystemDependentCharsIfNecessary(String path) {
        if (File.separatorChar == '\\') {
            return path.replace("\\", "\\\\");
        }
        return path;
    }

    public static String toSystemIndependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        return path;
    }

    public static String toExportableSystemDependentPath(File file2) {
        if (File.separatorChar != '/' && !file2.getAbsolutePath().startsWith("\\\\?\\")) {
            return "\\\\?\\" + file2.getAbsolutePath();
        }
        return file2.getAbsolutePath();
    }

    public static String sha1(File file2) throws IOException {
        return Hashing.sha1().hashBytes(com.google.common.io.Files.toByteArray(file2)).toString();
    }

    public static FluentIterable<File> getAllFiles(File dir) {
        return FluentIterable.from(com.google.common.io.Files.fileTraverser().depthFirstPreOrder(dir)).filter(com.google.common.io.Files.isFile());
    }

    public static String getNamesAsCommaSeparatedList(Iterable<File> files2) {
        return COMMA_SEPARATED_JOINER.join(Iterables.transform(files2, File::getName));
    }

    public static String sanitizeFileName(String input2) {
        return input2.replaceAll("[:\\\\/*\"?|<>']", "_");
    }

    public static String getDirectoryNameForJar(File inputFile) {
        HashFunction hashFunction2 = Hashing.sha1();
        HashCode hashCode = hashFunction2.hashString(inputFile.getAbsolutePath(), Charsets.UTF_16LE);
        Object name2 = com.google.common.io.Files.getNameWithoutExtension(inputFile.getName());
        if (((String)name2).equals("classes") && inputFile.getAbsolutePath().contains("exploded-aar")) {
            File versionDir = inputFile.getParentFile().getParentFile();
            File artifactDir = versionDir.getParentFile();
            File groupDir = artifactDir.getParentFile();
            name2 = Joiner.on('-').join(groupDir.getName(), artifactDir.getName(), versionDir.getName());
        }
        name2 = (String)name2 + "_" + hashCode.toString();
        return name2;
    }

    public static void createFile(File file2, String content) throws IOException {
        Preconditions.checkArgument(!file2.exists(), "%s exists already.", (Object)file2);
        FileUtils.writeToFile(file2, content);
    }

    public static void writeToFile(File file2, String content) throws IOException {
        com.google.common.io.Files.createParentDirs(file2);
        com.google.common.io.Files.asCharSink(file2, StandardCharsets.UTF_8, new FileWriteMode[0]).write(content);
    }

    public static List<File> find(File base, Pattern pattern) {
        Preconditions.checkArgument(base.isDirectory(), "'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from(com.google.common.io.Files.fileTraverser().depthFirstPreOrder(base)).filter(file2 -> pattern.matcher(FileUtils.toSystemIndependentPath(file2.getPath())).find()).toList();
    }

    public static Optional<File> find(File base, String name2) {
        Preconditions.checkArgument(base.isDirectory(), "'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from(com.google.common.io.Files.fileTraverser().depthFirstPreOrder(base)).filter(file2 -> name2.equals(file2.getName())).last();
    }

    public static String joinFilePaths(Iterable<File> files2) {
        return Joiner.on(File.pathSeparatorChar).join(Iterables.transform(files2, File::getAbsolutePath));
    }

    public static boolean parentDirExists(File file2) {
        File canonicalFile;
        try {
            canonicalFile = file2.getCanonicalFile();
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
        return canonicalFile.getParentFile() != null && canonicalFile.getParentFile().exists();
    }

    public static boolean isFileInDirectory(File fileOrDir, File baseDir) {
        String relativePath;
        try {
            relativePath = baseDir.toPath().relativize(fileOrDir.toPath()).toString();
        }
        catch (IllegalArgumentException e2) {
            return false;
        }
        return !relativePath.isEmpty() && !relativePath.startsWith("..");
    }

    public static boolean isSameFile(File file1, File file2) {
        try {
            if (file1.exists() && file2.exists()) {
                return Files.isSameFile(file1.toPath(), file2.toPath());
            }
            return file1.getCanonicalFile().equals(file2.getCanonicalFile());
        }
        catch (IOException e2) {
            throw new UncheckedIOException(e2);
        }
    }

    public static FileSystem createZipFilesystem(Path archive) throws IOException {
        URI uri = URI.create("jar:" + archive.toUri().toString());
        return FileSystems.newFileSystem(uri, Collections.emptyMap());
    }
}

