/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import androidx.annotation.Nullable;
import com.google.common.base.Preconditions;
import com.google.firebase.Timestamp;
import com.google.firebase.database.collection.ImmutableSortedSet;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.local.DocumentReference;
import com.google.firebase.firestore.local.LocalSerializer;
import com.google.firebase.firestore.local.MemoryPersistence;
import com.google.firebase.firestore.local.MutationQueue;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.ResourcePath;
import com.google.firebase.firestore.model.mutation.Mutation;
import com.google.firebase.firestore.model.mutation.MutationBatch;
import com.google.firebase.firestore.remote.WriteStream;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Util;
import com.google.protobuf.ByteString;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

final class MemoryMutationQueue
implements MutationQueue {
    private final List<MutationBatch> queue;
    private ImmutableSortedSet<DocumentReference> batchesByDocumentKey;
    private int nextBatchId;
    private ByteString lastStreamToken;
    private final MemoryPersistence persistence;

    MemoryMutationQueue(MemoryPersistence persistence) {
        this.persistence = persistence;
        this.queue = new ArrayList<MutationBatch>();
        this.batchesByDocumentKey = new ImmutableSortedSet(Collections.emptyList(), DocumentReference.BY_KEY);
        this.nextBatchId = 1;
        this.lastStreamToken = WriteStream.EMPTY_STREAM_TOKEN;
    }

    @Override
    public void start() {
        if (this.isEmpty()) {
            this.nextBatchId = 1;
        }
    }

    @Override
    public boolean isEmpty() {
        return this.queue.isEmpty();
    }

    @Override
    public void acknowledgeBatch(MutationBatch batch, ByteString streamToken) {
        int batchId = batch.getBatchId();
        int batchIndex = this.indexOfExistingBatchId(batchId, "acknowledged");
        Assert.hardAssert(batchIndex == 0, "Can only acknowledge the first batch in the mutation queue", new Object[0]);
        MutationBatch check = this.queue.get(batchIndex);
        Assert.hardAssert(batchId == check.getBatchId(), "Queue ordering failure: expected batch %d, got batch %d", batchId, check.getBatchId());
        this.lastStreamToken = (ByteString)Preconditions.checkNotNull((Object)streamToken);
    }

    @Override
    public ByteString getLastStreamToken() {
        return this.lastStreamToken;
    }

    @Override
    public void setLastStreamToken(ByteString streamToken) {
        this.lastStreamToken = (ByteString)Preconditions.checkNotNull((Object)streamToken);
    }

    @Override
    public MutationBatch addMutationBatch(Timestamp localWriteTime, List<Mutation> baseMutations, List<Mutation> mutations) {
        Assert.hardAssert(!mutations.isEmpty(), "Mutation batches should not be empty", new Object[0]);
        int batchId = this.nextBatchId++;
        int size = this.queue.size();
        if (size > 0) {
            MutationBatch prior = this.queue.get(size - 1);
            Assert.hardAssert(prior.getBatchId() < batchId, "Mutation batchIds must be monotonically increasing order", new Object[0]);
        }
        MutationBatch batch = new MutationBatch(batchId, localWriteTime, baseMutations, mutations);
        this.queue.add(batch);
        for (Mutation mutation : mutations) {
            this.batchesByDocumentKey = this.batchesByDocumentKey.insert((Object)new DocumentReference(mutation.getKey(), batchId));
            this.persistence.getIndexManager().addToCollectionParentIndex((ResourcePath)mutation.getKey().getPath().popLast());
        }
        return batch;
    }

    @Override
    @Nullable
    public MutationBatch lookupMutationBatch(int batchId) {
        int index = this.indexOfBatchId(batchId);
        if (index < 0 || index >= this.queue.size()) {
            return null;
        }
        MutationBatch batch = this.queue.get(index);
        Assert.hardAssert(batch.getBatchId() == batchId, "If found batch must match", new Object[0]);
        return batch;
    }

    @Override
    @Nullable
    public MutationBatch getNextMutationBatchAfterBatchId(int batchId) {
        int nextBatchId = batchId + 1;
        int rawIndex = this.indexOfBatchId(nextBatchId);
        int index = rawIndex < 0 ? 0 : rawIndex;
        return this.queue.size() > index ? this.queue.get(index) : null;
    }

    @Override
    public int getHighestUnacknowledgedBatchId() {
        return this.queue.isEmpty() ? -1 : this.nextBatchId - 1;
    }

    @Override
    public List<MutationBatch> getAllMutationBatches() {
        return Collections.unmodifiableList(this.queue);
    }

    @Override
    public List<MutationBatch> getAllMutationBatchesAffectingDocumentKey(DocumentKey documentKey) {
        DocumentReference reference;
        DocumentReference start = new DocumentReference(documentKey, 0);
        ArrayList<MutationBatch> result = new ArrayList<MutationBatch>();
        Iterator iterator = this.batchesByDocumentKey.iteratorFrom((Object)start);
        while (iterator.hasNext() && documentKey.equals((reference = (DocumentReference)iterator.next()).getKey())) {
            MutationBatch batch = this.lookupMutationBatch(reference.getId());
            Assert.hardAssert(batch != null, "Batches in the index must exist in the main table", new Object[0]);
            result.add(batch);
        }
        return result;
    }

    @Override
    public List<MutationBatch> getAllMutationBatchesAffectingDocumentKeys(Iterable<DocumentKey> documentKeys) {
        ImmutableSortedSet uniqueBatchIDs = new ImmutableSortedSet(Collections.emptyList(), Util.comparator());
        for (DocumentKey key : documentKeys) {
            DocumentReference reference;
            DocumentReference start = new DocumentReference(key, 0);
            Iterator batchesIter = this.batchesByDocumentKey.iteratorFrom((Object)start);
            while (batchesIter.hasNext() && key.equals((reference = (DocumentReference)batchesIter.next()).getKey())) {
                uniqueBatchIDs = uniqueBatchIDs.insert((Object)reference.getId());
            }
        }
        return this.lookupMutationBatches((ImmutableSortedSet<Integer>)uniqueBatchIDs);
    }

    @Override
    public List<MutationBatch> getAllMutationBatchesAffectingQuery(Query query) {
        DocumentReference reference;
        ResourcePath rowKeyPath;
        Assert.hardAssert(!query.isCollectionGroupQuery(), "CollectionGroup queries should be handled in LocalDocumentsView", new Object[0]);
        ResourcePath prefix = query.getPath();
        int immediateChildrenPathLength = prefix.length() + 1;
        ResourcePath startPath = prefix;
        if (!DocumentKey.isDocumentKey(startPath)) {
            startPath = (ResourcePath)((Object)startPath.append(""));
        }
        DocumentReference start = new DocumentReference(DocumentKey.fromPath(startPath), 0);
        ImmutableSortedSet uniqueBatchIDs = new ImmutableSortedSet(Collections.emptyList(), Util.comparator());
        Iterator iterator = this.batchesByDocumentKey.iteratorFrom((Object)start);
        while (iterator.hasNext() && prefix.isPrefixOf(rowKeyPath = (reference = (DocumentReference)iterator.next()).getKey().getPath())) {
            if (rowKeyPath.length() != immediateChildrenPathLength) continue;
            uniqueBatchIDs = uniqueBatchIDs.insert((Object)reference.getId());
        }
        return this.lookupMutationBatches((ImmutableSortedSet<Integer>)uniqueBatchIDs);
    }

    private List<MutationBatch> lookupMutationBatches(ImmutableSortedSet<Integer> batchIds) {
        ArrayList<MutationBatch> result = new ArrayList<MutationBatch>();
        for (Integer batchId : batchIds) {
            MutationBatch batch = this.lookupMutationBatch(batchId);
            if (batch == null) continue;
            result.add(batch);
        }
        return result;
    }

    @Override
    public void removeMutationBatch(MutationBatch batch) {
        int batchIndex = this.indexOfExistingBatchId(batch.getBatchId(), "removed");
        Assert.hardAssert(batchIndex == 0, "Can only remove the first entry of the mutation queue", new Object[0]);
        this.queue.remove(0);
        ImmutableSortedSet references = this.batchesByDocumentKey;
        for (Mutation mutation : batch.getMutations()) {
            DocumentKey key = mutation.getKey();
            this.persistence.getReferenceDelegate().removeMutationReference(key);
            DocumentReference reference = new DocumentReference(key, batch.getBatchId());
            references = references.remove((Object)reference);
        }
        this.batchesByDocumentKey = references;
    }

    @Override
    public void performConsistencyCheck() {
        if (this.queue.isEmpty()) {
            Assert.hardAssert(this.batchesByDocumentKey.isEmpty(), "Document leak -- detected dangling mutation references when queue is empty.", new Object[0]);
        }
    }

    boolean containsKey(DocumentKey key) {
        DocumentReference reference = new DocumentReference(key, 0);
        Iterator iterator = this.batchesByDocumentKey.iteratorFrom((Object)reference);
        if (!iterator.hasNext()) {
            return false;
        }
        DocumentKey firstKey = ((DocumentReference)iterator.next()).getKey();
        return firstKey.equals(key);
    }

    private int indexOfBatchId(int batchId) {
        if (this.queue.isEmpty()) {
            return 0;
        }
        MutationBatch firstBatch = this.queue.get(0);
        int firstBatchId = firstBatch.getBatchId();
        return batchId - firstBatchId;
    }

    private int indexOfExistingBatchId(int batchId, String action) {
        int index = this.indexOfBatchId(batchId);
        Assert.hardAssert(index >= 0 && index < this.queue.size(), "Batches must exist to be %s", action);
        return index;
    }

    long getByteSize(LocalSerializer serializer) {
        long count = 0L;
        for (MutationBatch batch : this.queue) {
            count += (long)serializer.encodeMutationBatch(batch).getSerializedSize();
        }
        return count;
    }
}

