/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.model;

import androidx.annotation.Nullable;
import com.google.firebase.firestore.model.DatabaseId;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.ServerTimestamps;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Util;
import com.google.firestore.v1.ArrayValue;
import com.google.firestore.v1.ArrayValueOrBuilder;
import com.google.firestore.v1.MapValue;
import com.google.firestore.v1.Value;
import com.google.protobuf.NullValue;
import com.google.protobuf.Timestamp;
import com.google.type.LatLng;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

public class Values {
    public static final Value NAN_VALUE = (Value)Value.newBuilder().setDoubleValue(Double.NaN).build();
    public static final Value NULL_VALUE = (Value)Value.newBuilder().setNullValue(NullValue.NULL_VALUE).build();
    public static final int TYPE_ORDER_NULL = 0;
    public static final int TYPE_ORDER_BOOLEAN = 1;
    public static final int TYPE_ORDER_NUMBER = 2;
    public static final int TYPE_ORDER_TIMESTAMP = 3;
    public static final int TYPE_ORDER_SERVER_TIMESTAMP = 4;
    public static final int TYPE_ORDER_STRING = 5;
    public static final int TYPE_ORDER_BLOB = 6;
    public static final int TYPE_ORDER_REFERENCE = 7;
    public static final int TYPE_ORDER_GEOPOINT = 8;
    public static final int TYPE_ORDER_ARRAY = 9;
    public static final int TYPE_ORDER_MAP = 10;

    public static int typeOrder(Value value) {
        switch (value.getValueTypeCase()) {
            case NULL_VALUE: {
                return 0;
            }
            case BOOLEAN_VALUE: {
                return 1;
            }
            case INTEGER_VALUE: {
                return 2;
            }
            case DOUBLE_VALUE: {
                return 2;
            }
            case TIMESTAMP_VALUE: {
                return 3;
            }
            case STRING_VALUE: {
                return 5;
            }
            case BYTES_VALUE: {
                return 6;
            }
            case REFERENCE_VALUE: {
                return 7;
            }
            case GEO_POINT_VALUE: {
                return 8;
            }
            case ARRAY_VALUE: {
                return 9;
            }
            case MAP_VALUE: {
                if (ServerTimestamps.isServerTimestamp(value)) {
                    return 4;
                }
                return 10;
            }
        }
        throw Assert.fail("Invalid value type: " + (Object)((Object)value.getValueTypeCase()), new Object[0]);
    }

    public static boolean equals(Value left, Value right) {
        int rightType;
        if (left == null && right == null) {
            return true;
        }
        if (left == null || right == null) {
            return false;
        }
        int leftType = Values.typeOrder(left);
        if (leftType != (rightType = Values.typeOrder(right))) {
            return false;
        }
        switch (leftType) {
            case 2: {
                return Values.numberEquals(left, right);
            }
            case 9: {
                return Values.arrayEquals(left, right);
            }
            case 10: {
                return Values.objectEquals(left, right);
            }
            case 4: {
                return ServerTimestamps.getLocalWriteTime(left).equals((Object)ServerTimestamps.getLocalWriteTime(right));
            }
        }
        return left.equals(right);
    }

    private static boolean numberEquals(Value left, Value right) {
        if (left.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE && right.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE) {
            return left.getIntegerValue() == right.getIntegerValue();
        }
        if (left.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE && right.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE) {
            return Double.doubleToLongBits(left.getDoubleValue()) == Double.doubleToLongBits(right.getDoubleValue());
        }
        return false;
    }

    private static boolean arrayEquals(Value left, Value right) {
        ArrayValue leftArray = left.getArrayValue();
        ArrayValue rightArray = right.getArrayValue();
        if (leftArray.getValuesCount() != rightArray.getValuesCount()) {
            return false;
        }
        for (int i = 0; i < leftArray.getValuesCount(); ++i) {
            if (Values.equals(leftArray.getValues(i), rightArray.getValues(i))) continue;
            return false;
        }
        return true;
    }

    private static boolean objectEquals(Value left, Value right) {
        MapValue leftMap = left.getMapValue();
        MapValue rightMap = right.getMapValue();
        if (leftMap.getFieldsCount() != rightMap.getFieldsCount()) {
            return false;
        }
        for (Map.Entry<String, Value> entry : leftMap.getFieldsMap().entrySet()) {
            Value otherEntry = rightMap.getFieldsMap().get(entry.getKey());
            if (Values.equals(entry.getValue(), otherEntry)) continue;
            return false;
        }
        return true;
    }

    public static boolean contains(ArrayValueOrBuilder haystack, Value needle) {
        for (Value haystackElement : haystack.getValuesList()) {
            if (!Values.equals(haystackElement, needle)) continue;
            return true;
        }
        return false;
    }

    public static int compare(Value left, Value right) {
        int rightType;
        int leftType = Values.typeOrder(left);
        if (leftType != (rightType = Values.typeOrder(right))) {
            return Util.compareIntegers(leftType, rightType);
        }
        switch (leftType) {
            case 0: {
                return 0;
            }
            case 1: {
                return Util.compareBooleans(left.getBooleanValue(), right.getBooleanValue());
            }
            case 2: {
                return Values.compareNumbers(left, right);
            }
            case 3: {
                return Values.compareTimestamps(left.getTimestampValue(), right.getTimestampValue());
            }
            case 4: {
                return Values.compareTimestamps(ServerTimestamps.getLocalWriteTime(left), ServerTimestamps.getLocalWriteTime(right));
            }
            case 5: {
                return left.getStringValue().compareTo(right.getStringValue());
            }
            case 6: {
                return Util.compareByteStrings(left.getBytesValue(), right.getBytesValue());
            }
            case 7: {
                return Values.compareReferences(left.getReferenceValue(), right.getReferenceValue());
            }
            case 8: {
                return Values.compareGeoPoints(left.getGeoPointValue(), right.getGeoPointValue());
            }
            case 9: {
                return Values.compareArrays(left.getArrayValue(), right.getArrayValue());
            }
            case 10: {
                return Values.compareMaps(left.getMapValue(), right.getMapValue());
            }
        }
        throw Assert.fail("Invalid value type: " + leftType, new Object[0]);
    }

    private static int compareNumbers(Value left, Value right) {
        if (left.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE) {
            double leftDouble = left.getDoubleValue();
            if (right.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE) {
                return Util.compareDoubles(leftDouble, right.getDoubleValue());
            }
            if (right.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE) {
                return Util.compareMixed(leftDouble, right.getIntegerValue());
            }
        } else if (left.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE) {
            long leftLong = left.getIntegerValue();
            if (right.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE) {
                return Util.compareLongs(leftLong, right.getIntegerValue());
            }
            if (right.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE) {
                return -1 * Util.compareMixed(right.getDoubleValue(), leftLong);
            }
        }
        throw Assert.fail("Unexpected values: %s vs %s", left, right);
    }

    private static int compareTimestamps(Timestamp left, Timestamp right) {
        int cmp = Util.compareLongs(left.getSeconds(), right.getSeconds());
        if (cmp != 0) {
            return cmp;
        }
        return Util.compareIntegers(left.getNanos(), right.getNanos());
    }

    private static int compareReferences(String leftPath, String rightPath) {
        String[] leftSegments = leftPath.split("/", -1);
        String[] rightSegments = rightPath.split("/", -1);
        int minLength = Math.min(leftSegments.length, rightSegments.length);
        for (int i = 0; i < minLength; ++i) {
            int cmp = leftSegments[i].compareTo(rightSegments[i]);
            if (cmp == 0) continue;
            return cmp;
        }
        return Util.compareIntegers(leftSegments.length, rightSegments.length);
    }

    private static int compareGeoPoints(LatLng left, LatLng right) {
        int comparison = Util.compareDoubles(left.getLatitude(), right.getLatitude());
        if (comparison == 0) {
            return Util.compareDoubles(left.getLongitude(), right.getLongitude());
        }
        return comparison;
    }

    private static int compareArrays(ArrayValue left, ArrayValue right) {
        int minLength = Math.min(left.getValuesCount(), right.getValuesCount());
        for (int i = 0; i < minLength; ++i) {
            int cmp = Values.compare(left.getValues(i), right.getValues(i));
            if (cmp == 0) continue;
            return cmp;
        }
        return Util.compareIntegers(left.getValuesCount(), right.getValuesCount());
    }

    private static int compareMaps(MapValue left, MapValue right) {
        Iterator<Map.Entry<String, Value>> iterator1 = new TreeMap<String, Value>(left.getFieldsMap()).entrySet().iterator();
        Iterator<Map.Entry<String, Value>> iterator2 = new TreeMap<String, Value>(right.getFieldsMap()).entrySet().iterator();
        while (iterator1.hasNext() && iterator2.hasNext()) {
            Map.Entry<String, Value> entry1 = iterator1.next();
            Map.Entry<String, Value> entry2 = iterator2.next();
            int keyCompare = entry1.getKey().compareTo(entry2.getKey());
            if (keyCompare != 0) {
                return keyCompare;
            }
            int valueCompare = Values.compare(entry1.getValue(), entry2.getValue());
            if (valueCompare == 0) continue;
            return valueCompare;
        }
        return Util.compareBooleans(iterator1.hasNext(), iterator2.hasNext());
    }

    public static String canonicalId(Value value) {
        StringBuilder builder = new StringBuilder();
        Values.canonifyValue(builder, value);
        return builder.toString();
    }

    private static void canonifyValue(StringBuilder builder, Value value) {
        switch (value.getValueTypeCase()) {
            case NULL_VALUE: {
                builder.append("null");
                break;
            }
            case BOOLEAN_VALUE: {
                builder.append(value.getBooleanValue());
                break;
            }
            case INTEGER_VALUE: {
                builder.append(value.getIntegerValue());
                break;
            }
            case DOUBLE_VALUE: {
                builder.append(value.getDoubleValue());
                break;
            }
            case TIMESTAMP_VALUE: {
                Values.canonifyTimestamp(builder, value.getTimestampValue());
                break;
            }
            case STRING_VALUE: {
                builder.append(value.getStringValue());
                break;
            }
            case BYTES_VALUE: {
                builder.append(Util.toDebugString(value.getBytesValue()));
                break;
            }
            case REFERENCE_VALUE: {
                Values.canonifyReference(builder, value);
                break;
            }
            case GEO_POINT_VALUE: {
                Values.canonifyGeoPoint(builder, value.getGeoPointValue());
                break;
            }
            case ARRAY_VALUE: {
                Values.canonifyArray(builder, value.getArrayValue());
                break;
            }
            case MAP_VALUE: {
                Values.canonifyObject(builder, value.getMapValue());
                break;
            }
            default: {
                throw Assert.fail("Invalid value type: " + (Object)((Object)value.getValueTypeCase()), new Object[0]);
            }
        }
    }

    private static void canonifyTimestamp(StringBuilder builder, Timestamp timestamp) {
        builder.append(String.format("time(%s,%s)", timestamp.getSeconds(), timestamp.getNanos()));
    }

    private static void canonifyGeoPoint(StringBuilder builder, LatLng latLng) {
        builder.append(String.format("geo(%s,%s)", latLng.getLatitude(), latLng.getLongitude()));
    }

    private static void canonifyReference(StringBuilder builder, Value value) {
        Assert.hardAssert(Values.isReferenceValue(value), "Value should be a ReferenceValue", new Object[0]);
        builder.append(DocumentKey.fromName(value.getReferenceValue()));
    }

    private static void canonifyObject(StringBuilder builder, MapValue mapValue) {
        ArrayList<String> keys = new ArrayList<String>(mapValue.getFieldsMap().keySet());
        Collections.sort(keys);
        builder.append("{");
        boolean first = true;
        for (String key : keys) {
            if (!first) {
                builder.append(",");
            } else {
                first = false;
            }
            builder.append(key).append(":");
            Values.canonifyValue(builder, mapValue.getFieldsOrThrow(key));
        }
        builder.append("}");
    }

    private static void canonifyArray(StringBuilder builder, ArrayValue arrayValue) {
        builder.append("[");
        for (int i = 0; i < arrayValue.getValuesCount(); ++i) {
            Values.canonifyValue(builder, arrayValue.getValues(i));
            if (i == arrayValue.getValuesCount() - 1) continue;
            builder.append(",");
        }
        builder.append("]");
    }

    public static boolean isInteger(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.INTEGER_VALUE;
    }

    public static boolean isDouble(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.DOUBLE_VALUE;
    }

    public static boolean isNumber(@Nullable Value value) {
        return Values.isInteger(value) || Values.isDouble(value);
    }

    public static boolean isArray(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.ARRAY_VALUE;
    }

    public static boolean isReferenceValue(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.REFERENCE_VALUE;
    }

    public static boolean isNullValue(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.NULL_VALUE;
    }

    public static boolean isNanValue(@Nullable Value value) {
        return value != null && Double.isNaN(value.getDoubleValue());
    }

    public static boolean isMapValue(@Nullable Value value) {
        return value != null && value.getValueTypeCase() == Value.ValueTypeCase.MAP_VALUE;
    }

    public static Value refValue(DatabaseId databaseId, DocumentKey key) {
        Value value = (Value)Value.newBuilder().setReferenceValue(String.format("projects/%s/databases/%s/documents/%s", databaseId.getProjectId(), databaseId.getDatabaseId(), key.toString())).build();
        return value;
    }
}

