/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import android.util.SparseArray;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import com.google.firebase.Timestamp;
import com.google.firebase.database.collection.ImmutableSortedMap;
import com.google.firebase.database.collection.ImmutableSortedSet;
import com.google.firebase.firestore.auth.User;
import com.google.firebase.firestore.bundle.BundleCallback;
import com.google.firebase.firestore.bundle.BundleMetadata;
import com.google.firebase.firestore.bundle.NamedQuery;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.core.Target;
import com.google.firebase.firestore.core.TargetIdGenerator;
import com.google.firebase.firestore.local.BundleCache;
import com.google.firebase.firestore.local.DocumentOverlayCache;
import com.google.firebase.firestore.local.IndexManager;
import com.google.firebase.firestore.local.LocalDocumentsResult;
import com.google.firebase.firestore.local.LocalDocumentsView;
import com.google.firebase.firestore.local.LocalViewChanges;
import com.google.firebase.firestore.local.LruGarbageCollector;
import com.google.firebase.firestore.local.MutationQueue;
import com.google.firebase.firestore.local.OverlayedDocument;
import com.google.firebase.firestore.local.Persistence;
import com.google.firebase.firestore.local.QueryEngine;
import com.google.firebase.firestore.local.QueryPurpose;
import com.google.firebase.firestore.local.QueryResult;
import com.google.firebase.firestore.local.ReferenceSet;
import com.google.firebase.firestore.local.RemoteDocumentCache;
import com.google.firebase.firestore.local.TargetCache;
import com.google.firebase.firestore.local.TargetData;
import com.google.firebase.firestore.model.Document;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.FieldIndex;
import com.google.firebase.firestore.model.MutableDocument;
import com.google.firebase.firestore.model.ObjectValue;
import com.google.firebase.firestore.model.ResourcePath;
import com.google.firebase.firestore.model.SnapshotVersion;
import com.google.firebase.firestore.model.mutation.Mutation;
import com.google.firebase.firestore.model.mutation.MutationBatch;
import com.google.firebase.firestore.model.mutation.MutationBatchResult;
import com.google.firebase.firestore.model.mutation.MutationResult;
import com.google.firebase.firestore.model.mutation.PatchMutation;
import com.google.firebase.firestore.model.mutation.Precondition;
import com.google.firebase.firestore.remote.RemoteEvent;
import com.google.firebase.firestore.remote.TargetChange;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Logger;
import com.google.firebase.firestore.util.Util;
import com.google.protobuf.ByteString;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public final class LocalStore
implements BundleCallback {
    private static final long RESUME_TOKEN_MAX_AGE_SECONDS = TimeUnit.MINUTES.toSeconds(5L);
    private final Persistence persistence;
    private IndexManager indexManager;
    private MutationQueue mutationQueue;
    private DocumentOverlayCache documentOverlayCache;
    private final RemoteDocumentCache remoteDocuments;
    private LocalDocumentsView localDocuments;
    private final QueryEngine queryEngine;
    private final ReferenceSet localViewReferences;
    private final TargetCache targetCache;
    private final BundleCache bundleCache;
    private final SparseArray<TargetData> queryDataByTarget;
    private final Map<Target, Integer> targetIdByTarget;
    private final TargetIdGenerator targetIdGenerator;

    public LocalStore(Persistence persistence, QueryEngine queryEngine, User initialUser) {
        Assert.hardAssert(persistence.isStarted(), "LocalStore was passed an unstarted persistence implementation", new Object[0]);
        this.persistence = persistence;
        this.queryEngine = queryEngine;
        this.targetCache = persistence.getTargetCache();
        this.bundleCache = persistence.getBundleCache();
        this.targetIdGenerator = TargetIdGenerator.forTargetCache(this.targetCache.getHighestTargetId());
        this.remoteDocuments = persistence.getRemoteDocumentCache();
        this.localViewReferences = new ReferenceSet();
        this.queryDataByTarget = new SparseArray();
        this.targetIdByTarget = new HashMap<Target, Integer>();
        persistence.getReferenceDelegate().setInMemoryPins(this.localViewReferences);
        this.initializeUserComponents(initialUser);
    }

    private void initializeUserComponents(User user) {
        this.indexManager = this.persistence.getIndexManager(user);
        this.mutationQueue = this.persistence.getMutationQueue(user, this.indexManager);
        this.documentOverlayCache = this.persistence.getDocumentOverlayCache(user);
        this.localDocuments = new LocalDocumentsView(this.remoteDocuments, this.mutationQueue, this.documentOverlayCache, this.indexManager);
        this.remoteDocuments.setIndexManager(this.indexManager);
        this.queryEngine.initialize(this.localDocuments, this.indexManager);
    }

    public void start() {
        this.persistence.getOverlayMigrationManager().run();
        this.startIndexManager();
        this.startMutationQueue();
    }

    private void startIndexManager() {
        this.persistence.runTransaction("Start IndexManager", () -> this.indexManager.start());
    }

    private void startMutationQueue() {
        this.persistence.runTransaction("Start MutationQueue", () -> this.mutationQueue.start());
    }

    public IndexManager getIndexManagerForCurrentUser() {
        return this.indexManager;
    }

    public LocalDocumentsView getLocalDocumentsForCurrentUser() {
        return this.localDocuments;
    }

    public ImmutableSortedMap<DocumentKey, Document> handleUserChange(User user) {
        List<MutationBatch> oldBatches = this.mutationQueue.getAllMutationBatches();
        this.initializeUserComponents(user);
        this.startIndexManager();
        this.startMutationQueue();
        List<MutationBatch> newBatches = this.mutationQueue.getAllMutationBatches();
        ImmutableSortedSet changedKeys = DocumentKey.emptyKeySet();
        for (List batches : Arrays.asList(oldBatches, newBatches)) {
            for (MutationBatch batch : batches) {
                for (Mutation mutation : batch.getMutations()) {
                    changedKeys = changedKeys.insert((Object)mutation.getKey());
                }
            }
        }
        return this.localDocuments.getDocuments((Iterable<DocumentKey>)changedKeys);
    }

    public LocalDocumentsResult writeLocally(List<Mutation> mutations) {
        Timestamp localWriteTime = Timestamp.now();
        HashSet<DocumentKey> keys = new HashSet<DocumentKey>();
        for (Mutation mutation : mutations) {
            keys.add(mutation.getKey());
        }
        return this.persistence.runTransaction("Locally write mutations", () -> {
            Map<DocumentKey, MutableDocument> remoteDocs = this.remoteDocuments.getAll(keys);
            HashSet<DocumentKey> docsWithoutRemoteVersion = new HashSet<DocumentKey>();
            for (Map.Entry<DocumentKey, MutableDocument> entry : remoteDocs.entrySet()) {
                if (entry.getValue().isValidDocument()) continue;
                docsWithoutRemoteVersion.add(entry.getKey());
            }
            Map<DocumentKey, OverlayedDocument> overlayedDocuments = this.localDocuments.getOverlayedDocuments(remoteDocs);
            ArrayList<Mutation> baseMutations = new ArrayList<Mutation>();
            for (Mutation mutation : mutations) {
                ObjectValue baseValue = mutation.extractTransformBaseValue(overlayedDocuments.get(mutation.getKey()).getDocument());
                if (baseValue == null) continue;
                baseMutations.add(new PatchMutation(mutation.getKey(), baseValue, baseValue.getFieldMask(), Precondition.exists(true)));
            }
            MutationBatch batch = this.mutationQueue.addMutationBatch(localWriteTime, baseMutations, mutations);
            Map<DocumentKey, Mutation> overlays = batch.applyToLocalDocumentSet(overlayedDocuments, docsWithoutRemoteVersion);
            this.documentOverlayCache.saveOverlays(batch.getBatchId(), overlays);
            return LocalDocumentsResult.fromOverlayedDocuments(batch.getBatchId(), overlayedDocuments);
        });
    }

    public ImmutableSortedMap<DocumentKey, Document> acknowledgeBatch(MutationBatchResult batchResult) {
        return this.persistence.runTransaction("Acknowledge batch", () -> {
            MutationBatch batch = batchResult.getBatch();
            this.mutationQueue.acknowledgeBatch(batch, batchResult.getStreamToken());
            this.applyWriteToRemoteDocuments(batchResult);
            this.mutationQueue.performConsistencyCheck();
            this.documentOverlayCache.removeOverlaysForBatchId(batchResult.getBatch().getBatchId());
            this.localDocuments.recalculateAndSaveOverlays(this.getKeysWithTransformResults(batchResult));
            return this.localDocuments.getDocuments(batch.getKeys());
        });
    }

    @NonNull
    private Set<DocumentKey> getKeysWithTransformResults(MutationBatchResult batchResult) {
        HashSet<DocumentKey> result = new HashSet<DocumentKey>();
        for (int i = 0; i < batchResult.getMutationResults().size(); ++i) {
            MutationResult mutationResult = batchResult.getMutationResults().get(i);
            if (mutationResult.getTransformResults().isEmpty()) continue;
            result.add(batchResult.getBatch().getMutations().get(i).getKey());
        }
        return result;
    }

    public ImmutableSortedMap<DocumentKey, Document> rejectBatch(int batchId) {
        return this.persistence.runTransaction("Reject batch", () -> {
            MutationBatch toReject = this.mutationQueue.lookupMutationBatch(batchId);
            Assert.hardAssert(toReject != null, "Attempt to reject nonexistent batch!", new Object[0]);
            this.mutationQueue.removeMutationBatch(toReject);
            this.mutationQueue.performConsistencyCheck();
            this.documentOverlayCache.removeOverlaysForBatchId(batchId);
            this.localDocuments.recalculateAndSaveOverlays(toReject.getKeys());
            return this.localDocuments.getDocuments(toReject.getKeys());
        });
    }

    public int getHighestUnacknowledgedBatchId() {
        return this.mutationQueue.getHighestUnacknowledgedBatchId();
    }

    public ByteString getLastStreamToken() {
        return this.mutationQueue.getLastStreamToken();
    }

    public void setLastStreamToken(ByteString streamToken) {
        this.persistence.runTransaction("Set stream token", () -> this.mutationQueue.setLastStreamToken(streamToken));
    }

    public SnapshotVersion getLastRemoteSnapshotVersion() {
        return this.targetCache.getLastRemoteSnapshotVersion();
    }

    public ImmutableSortedMap<DocumentKey, Document> applyRemoteEvent(RemoteEvent remoteEvent) {
        SnapshotVersion remoteVersion = remoteEvent.getSnapshotVersion();
        return this.persistence.runTransaction("Apply remote event", () -> {
            Map<Integer, TargetChange> targetChanges = remoteEvent.getTargetChanges();
            long sequenceNumber = this.persistence.getReferenceDelegate().getCurrentSequenceNumber();
            for (Map.Entry<Integer, TargetChange> entry : targetChanges.entrySet()) {
                Integer boxedTargetId = entry.getKey();
                int targetId = boxedTargetId;
                TargetChange change = entry.getValue();
                TargetData oldTargetData = (TargetData)this.queryDataByTarget.get(targetId);
                if (oldTargetData == null) continue;
                this.targetCache.removeMatchingKeys(change.getRemovedDocuments(), targetId);
                this.targetCache.addMatchingKeys(change.getAddedDocuments(), targetId);
                TargetData newTargetData = oldTargetData.withSequenceNumber(sequenceNumber);
                if (remoteEvent.getTargetMismatches().containsKey(targetId)) {
                    newTargetData = newTargetData.withResumeToken(ByteString.EMPTY, SnapshotVersion.NONE).withLastLimboFreeSnapshotVersion(SnapshotVersion.NONE);
                } else if (!change.getResumeToken().isEmpty()) {
                    newTargetData = newTargetData.withResumeToken(change.getResumeToken(), remoteEvent.getSnapshotVersion());
                }
                this.queryDataByTarget.put(targetId, (Object)newTargetData);
                if (!LocalStore.shouldPersistTargetData(oldTargetData, newTargetData, change)) continue;
                this.targetCache.updateTargetData(newTargetData);
            }
            Map<DocumentKey, MutableDocument> documentUpdates = remoteEvent.getDocumentUpdates();
            Set<DocumentKey> limboDocuments = remoteEvent.getResolvedLimboDocuments();
            for (DocumentKey key : documentUpdates.keySet()) {
                if (!limboDocuments.contains(key)) continue;
                this.persistence.getReferenceDelegate().updateLimboDocument(key);
            }
            DocumentChangeResult result = this.populateDocumentChanges(documentUpdates);
            Map changedDocs = result.changedDocuments;
            SnapshotVersion lastRemoteVersion = this.targetCache.getLastRemoteSnapshotVersion();
            if (!remoteVersion.equals(SnapshotVersion.NONE)) {
                Assert.hardAssert(remoteVersion.compareTo(lastRemoteVersion) >= 0, "Watch stream reverted to previous snapshot?? (%s < %s)", remoteVersion, lastRemoteVersion);
                this.targetCache.setLastRemoteSnapshotVersion(remoteVersion);
            }
            return this.localDocuments.getLocalViewOfDocuments(changedDocs, result.existenceChangedKeys);
        });
    }

    private DocumentChangeResult populateDocumentChanges(Map<DocumentKey, MutableDocument> documents) {
        HashMap<DocumentKey, MutableDocument> changedDocs = new HashMap<DocumentKey, MutableDocument>();
        ArrayList<DocumentKey> removedDocs = new ArrayList<DocumentKey>();
        HashSet<DocumentKey> conditionChanged = new HashSet<DocumentKey>();
        Map<DocumentKey, MutableDocument> existingDocs = this.remoteDocuments.getAll(documents.keySet());
        for (Map.Entry<DocumentKey, MutableDocument> entry : documents.entrySet()) {
            DocumentKey key = entry.getKey();
            MutableDocument doc = entry.getValue();
            MutableDocument existingDoc = existingDocs.get(key);
            if (doc.isFoundDocument() != existingDoc.isFoundDocument()) {
                conditionChanged.add(key);
            }
            if (doc.isNoDocument() && doc.getVersion().equals(SnapshotVersion.NONE)) {
                removedDocs.add(doc.getKey());
                changedDocs.put(key, doc);
                continue;
            }
            if (!existingDoc.isValidDocument() || doc.getVersion().compareTo(existingDoc.getVersion()) > 0 || doc.getVersion().compareTo(existingDoc.getVersion()) == 0 && existingDoc.hasPendingWrites()) {
                Assert.hardAssert(!SnapshotVersion.NONE.equals(doc.getReadTime()), "Cannot add a document when the remote version is zero", new Object[0]);
                this.remoteDocuments.add(doc, doc.getReadTime());
                changedDocs.put(key, doc);
                continue;
            }
            Logger.debug("LocalStore", "Ignoring outdated watch update for %s.Current version: %s  Watch version: %s", key, existingDoc.getVersion(), doc.getVersion());
        }
        this.remoteDocuments.removeAll(removedDocs);
        return new DocumentChangeResult(changedDocs, conditionChanged);
    }

    private static boolean shouldPersistTargetData(TargetData oldTargetData, TargetData newTargetData, @Nullable TargetChange change) {
        long oldLimboFreeSeconds;
        long oldSeconds;
        if (oldTargetData.getResumeToken().isEmpty()) {
            return true;
        }
        long newSeconds = newTargetData.getSnapshotVersion().getTimestamp().getSeconds();
        long timeDelta = newSeconds - (oldSeconds = oldTargetData.getSnapshotVersion().getTimestamp().getSeconds());
        if (timeDelta >= RESUME_TOKEN_MAX_AGE_SECONDS) {
            return true;
        }
        long newLimboFreeSeconds = newTargetData.getLastLimboFreeSnapshotVersion().getTimestamp().getSeconds();
        long limboFreeTimeDelta = newLimboFreeSeconds - (oldLimboFreeSeconds = oldTargetData.getLastLimboFreeSnapshotVersion().getTimestamp().getSeconds());
        if (limboFreeTimeDelta >= RESUME_TOKEN_MAX_AGE_SECONDS) {
            return true;
        }
        if (change == null) {
            return false;
        }
        int changes = change.getAddedDocuments().size() + change.getModifiedDocuments().size() + change.getRemovedDocuments().size();
        return changes > 0;
    }

    public void notifyLocalViewChanges(List<LocalViewChanges> viewChanges) {
        this.persistence.runTransaction("notifyLocalViewChanges", () -> {
            for (LocalViewChanges viewChange : viewChanges) {
                int targetId = viewChange.getTargetId();
                this.localViewReferences.addReferences(viewChange.getAdded(), targetId);
                ImmutableSortedSet<DocumentKey> removed = viewChange.getRemoved();
                for (DocumentKey key : removed) {
                    this.persistence.getReferenceDelegate().removeReference(key);
                }
                this.localViewReferences.removeReferences(removed, targetId);
                if (viewChange.isFromCache()) continue;
                TargetData targetData = (TargetData)this.queryDataByTarget.get(targetId);
                Assert.hardAssert(targetData != null, "Can't set limbo-free snapshot version for unknown target: %s", targetId);
                SnapshotVersion lastLimboFreeSnapshotVersion = targetData.getSnapshotVersion();
                TargetData updatedTargetData = targetData.withLastLimboFreeSnapshotVersion(lastLimboFreeSnapshotVersion);
                this.queryDataByTarget.put(targetId, (Object)updatedTargetData);
                if (!LocalStore.shouldPersistTargetData(targetData, updatedTargetData, null)) continue;
                this.targetCache.updateTargetData(updatedTargetData);
            }
        });
    }

    @Nullable
    public MutationBatch getNextMutationBatch(int afterBatchId) {
        return this.mutationQueue.getNextMutationBatchAfterBatchId(afterBatchId);
    }

    public Document readDocument(DocumentKey key) {
        return this.localDocuments.getDocument(key);
    }

    public TargetData allocateTarget(Target target) {
        int targetId;
        TargetData cached = this.targetCache.getTargetData(target);
        if (cached != null) {
            targetId = cached.getTargetId();
        } else {
            AllocateQueryHolder holder = new AllocateQueryHolder();
            this.persistence.runTransaction("Allocate target", () -> {
                holder.targetId = this.targetIdGenerator.nextId();
                holder.cached = new TargetData(target, holder.targetId, this.persistence.getReferenceDelegate().getCurrentSequenceNumber(), QueryPurpose.LISTEN);
                this.targetCache.addTargetData(holder.cached);
            });
            targetId = holder.targetId;
            cached = holder.cached;
        }
        if (this.queryDataByTarget.get(targetId) == null) {
            this.queryDataByTarget.put(targetId, (Object)cached);
            this.targetIdByTarget.put(target, targetId);
        }
        return cached;
    }

    @VisibleForTesting
    @Nullable
    TargetData getTargetData(Target target) {
        Integer targetId = this.targetIdByTarget.get(target);
        if (targetId != null) {
            return (TargetData)this.queryDataByTarget.get(targetId.intValue());
        }
        return this.targetCache.getTargetData(target);
    }

    public boolean hasNewerBundle(BundleMetadata bundleMetadata) {
        return this.persistence.runTransaction("Has newer bundle", () -> {
            BundleMetadata cachedMetadata = this.bundleCache.getBundleMetadata(bundleMetadata.getBundleId());
            return cachedMetadata != null && cachedMetadata.getCreateTime().compareTo(bundleMetadata.getCreateTime()) >= 0;
        });
    }

    @Override
    public void saveBundle(BundleMetadata bundleMetadata) {
        this.persistence.runTransaction("Save bundle", () -> this.bundleCache.saveBundleMetadata(bundleMetadata));
    }

    @Override
    public ImmutableSortedMap<DocumentKey, Document> applyBundledDocuments(ImmutableSortedMap<DocumentKey, MutableDocument> documents, String bundleId) {
        TargetData umbrellaTargetData = this.allocateTarget(LocalStore.newUmbrellaTarget(bundleId));
        return this.persistence.runTransaction("Apply bundle documents", () -> {
            ImmutableSortedSet documentKeys = DocumentKey.emptyKeySet();
            HashMap<DocumentKey, MutableDocument> documentMap = new HashMap<DocumentKey, MutableDocument>();
            for (Map.Entry entry : documents) {
                DocumentKey documentKey = (DocumentKey)entry.getKey();
                MutableDocument document = (MutableDocument)entry.getValue();
                if (document.isFoundDocument()) {
                    documentKeys = documentKeys.insert((Object)documentKey);
                }
                documentMap.put(documentKey, document);
            }
            this.targetCache.removeMatchingKeysForTargetId(umbrellaTargetData.getTargetId());
            this.targetCache.addMatchingKeys(documentKeys, umbrellaTargetData.getTargetId());
            DocumentChangeResult result = this.populateDocumentChanges(documentMap);
            Map changedDocs = result.changedDocuments;
            return this.localDocuments.getLocalViewOfDocuments(changedDocs, result.existenceChangedKeys);
        });
    }

    @Override
    public void saveNamedQuery(NamedQuery namedQuery, ImmutableSortedSet<DocumentKey> documentKeys) {
        TargetData existingTargetData = this.allocateTarget(namedQuery.getBundledQuery().getTarget());
        int targetId = existingTargetData.getTargetId();
        this.persistence.runTransaction("Saved named query", () -> {
            if (namedQuery.getReadTime().compareTo(existingTargetData.getSnapshotVersion()) > 0) {
                TargetData newTargetData = existingTargetData.withResumeToken(ByteString.EMPTY, namedQuery.getReadTime());
                this.queryDataByTarget.append(targetId, (Object)newTargetData);
                this.targetCache.updateTargetData(newTargetData);
                this.targetCache.removeMatchingKeysForTargetId(targetId);
                this.targetCache.addMatchingKeys(documentKeys, targetId);
            }
            this.bundleCache.saveNamedQuery(namedQuery);
        });
    }

    @Nullable
    public NamedQuery getNamedQuery(String queryName) {
        return this.persistence.runTransaction("Get named query", () -> this.bundleCache.getNamedQuery(queryName));
    }

    @VisibleForTesting
    Collection<FieldIndex> getFieldIndexes() {
        return this.persistence.runTransaction("Get indexes", () -> this.indexManager.getFieldIndexes());
    }

    public void configureFieldIndexes(List<FieldIndex> newFieldIndexes) {
        this.persistence.runTransaction("Configure indexes", () -> Util.diffCollections(this.indexManager.getFieldIndexes(), newFieldIndexes, FieldIndex.SEMANTIC_COMPARATOR, this.indexManager::addFieldIndex, this.indexManager::deleteFieldIndex));
    }

    public void deleteAllFieldIndexes() {
        this.persistence.runTransaction("Delete All Indexes", () -> this.indexManager.deleteAllFieldIndexes());
    }

    public void setIndexAutoCreationEnabled(boolean isEnabled) {
        this.queryEngine.setIndexAutoCreationEnabled(isEnabled);
    }

    public void releaseTarget(int targetId) {
        this.persistence.runTransaction("Release target", () -> {
            TargetData targetData = (TargetData)this.queryDataByTarget.get(targetId);
            Assert.hardAssert(targetData != null, "Tried to release nonexistent target: %s", targetId);
            ImmutableSortedSet<DocumentKey> removedReferences = this.localViewReferences.removeReferencesForId(targetId);
            for (DocumentKey key : removedReferences) {
                this.persistence.getReferenceDelegate().removeReference(key);
            }
            this.persistence.getReferenceDelegate().removeTarget(targetData);
            this.queryDataByTarget.remove(targetId);
            this.targetIdByTarget.remove(targetData.getTarget());
        });
    }

    public QueryResult executeQuery(Query query, boolean usePreviousResults) {
        TargetData targetData = this.getTargetData(query.toTarget());
        SnapshotVersion lastLimboFreeSnapshotVersion = SnapshotVersion.NONE;
        ImmutableSortedSet<DocumentKey> remoteKeys = DocumentKey.emptyKeySet();
        if (targetData != null) {
            lastLimboFreeSnapshotVersion = targetData.getLastLimboFreeSnapshotVersion();
            remoteKeys = this.targetCache.getMatchingKeysForTargetId(targetData.getTargetId());
        }
        ImmutableSortedMap<DocumentKey, Document> documents = this.queryEngine.getDocumentsMatchingQuery(query, usePreviousResults ? lastLimboFreeSnapshotVersion : SnapshotVersion.NONE, remoteKeys);
        return new QueryResult(documents, remoteKeys);
    }

    public ImmutableSortedSet<DocumentKey> getRemoteDocumentKeys(int targetId) {
        return this.targetCache.getMatchingKeysForTargetId(targetId);
    }

    private void applyWriteToRemoteDocuments(MutationBatchResult batchResult) {
        MutationBatch batch = batchResult.getBatch();
        Set<DocumentKey> docKeys = batch.getKeys();
        for (DocumentKey docKey : docKeys) {
            MutableDocument doc = this.remoteDocuments.get(docKey);
            SnapshotVersion ackVersion = (SnapshotVersion)batchResult.getDocVersions().get((Object)docKey);
            Assert.hardAssert(ackVersion != null, "docVersions should contain every doc in the write.", new Object[0]);
            if (doc.getVersion().compareTo(ackVersion) >= 0) continue;
            batch.applyToRemoteDocument(doc, batchResult);
            if (!doc.isValidDocument()) continue;
            this.remoteDocuments.add(doc, batchResult.getCommitVersion());
        }
        this.mutationQueue.removeMutationBatch(batch);
    }

    public LruGarbageCollector.Results collectGarbage(LruGarbageCollector garbageCollector) {
        return this.persistence.runTransaction("Collect garbage", () -> garbageCollector.collect(this.queryDataByTarget));
    }

    private static Target newUmbrellaTarget(String bundleName) {
        return Query.atPath(ResourcePath.fromString("__bundle__/docs/" + bundleName)).toTarget();
    }

    private static class AllocateQueryHolder {
        TargetData cached;
        int targetId;

        private AllocateQueryHolder() {
        }
    }

    private static class DocumentChangeResult {
        private final Map<DocumentKey, MutableDocument> changedDocuments;
        private final Set<DocumentKey> existenceChangedKeys;

        private DocumentChangeResult(Map<DocumentKey, MutableDocument> changedDocuments, Set<DocumentKey> existenceChangedKeys) {
            this.changedDocuments = changedDocuments;
            this.existenceChangedKeys = existenceChangedKeys;
        }
    }
}

