/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.model.apidoc.misc;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Optional;
import java.util.*;

public class TypeLinkCreator {

    private static TypeLinkCreator typeLinkCreator;
    private Map<String, String> packageUrl = new HashMap<>();

    private TypeLinkCreator(){}

    public static TypeLinkCreator getInstance() {
        if (typeLinkCreator == null) {
            typeLinkCreator = new TypeLinkCreator();
        }
        return typeLinkCreator;
    }

    public String create(String fullQualifiedName){

        List<String> generics;
        String pattern;
        if (fullQualifiedName.contains("<")) {
            pattern = fullQualifiedName.substring(0, fullQualifiedName.indexOf("<"));
            generics = genericsFromFullQualifiedName(fullQualifiedName);
        } else {
            pattern = fullQualifiedName;
            generics = new ArrayList<>();
        }

        String baseName = pattern.substring(pattern.lastIndexOf(".") + 1);
        StringBuilder builder = new StringBuilder();
        Optional<String> key = packageKeyMatch(pattern);
        if(key.isPresent()) {
            String baseUrl = packageUrl.get(key.get());
            String url = baseUrl.endsWith("/") ? baseUrl : baseUrl + "/";
            builder.append("<a href=\"" + url + pattern.replace(".","/") + ".html\">" + baseName + "</a>");
        } else {
            builder.append(baseName);
        }

        if (!generics.isEmpty()) {
            builder.append("<");
            Iterator<String> iterator = generics.iterator();
            builder.append(create(iterator.next()));
            while (iterator.hasNext()) {
                builder.append(",").append(create(iterator.next()));
            }
            builder.append(">");
        }
        return builder.toString();
    }

    private List<String> genericsFromFullQualifiedName(String fullQualifiedName) {
        List<String> generics = new ArrayList<>();
        if(fullQualifiedName.contains("<")) {
            String genericTypes = fullQualifiedName.substring(fullQualifiedName.indexOf("<") + 1, fullQualifiedName.lastIndexOf(">"));
            generics.addAll(parseTypes(genericTypes));
        }
        return generics;
    }

    private List<String> parseTypes(String genericTypes) {
        List<String> types = new ArrayList<>();
        int level = 0;
        int begin = 0;
        char[] chars = genericTypes.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            char c = chars[i];
            if(c == '<') {
                level ++;
            }
            if(c == '>') {
                level --;
            }
            if (c == ',' && level == 0) {
                types.add(genericTypes.substring(begin, i));
                begin = i + 1;
            }
        }
        types.add(genericTypes.substring(begin, chars.length));
        return types;
    }

    private Optional<String> packageKeyMatch(String fullQualifiedName) {
        for (String key : packageUrl.keySet()) {
            if (fullQualifiedName.startsWith(key)) {
                return Optional.fromNullable(key);
            }
        }
        return Optional.absent();
    }

    /**
     * Parses the @javadocUrl javadoc tag from the Module/Connector class javadoc and
     * following the convention: packageName[urlForThatPackage];packageName1[urlForThatPackage2];...
     * add entries to the packageUrl map.
     * @param javadocUrls the @javadocUrl tag from the Module/Connector class.
     */
    public void putEntriesFromJavadocTag(String javadocUrls){
        if (StringUtils.isNotBlank(javadocUrls)) {
            for (String s : javadocUrls.split(";")) {
                String _package = s.substring(0, s.indexOf("["));
                String url = s.substring(s.indexOf("[") + 1, s.indexOf("]"));
                packageUrl.put(_package, url);
            }
        }
    }

    /**
     * Put an entry in the packageUrl static map
     * @param _package the package key
     * @param url url for the given package key
     */
    public void putPackageUrl(String _package, String url) {
        packageUrl.put(_package, url);
    }

}
