/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.ext.saml2mdrpi.impl;

import javax.annotation.Nonnull;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.saml.common.AbstractSAMLObjectUnmarshaller;
import org.opensaml.saml.ext.saml2mdrpi.PublicationInfo;
import org.opensaml.saml.ext.saml2mdrpi.UsagePolicy;
import org.w3c.dom.Attr;

import net.shibboleth.shared.xml.DOMTypeSupport;

/**
 * An unmarshaller for {@link PublicationInfo}.
 */
public class PublicationInfoUnmarshaller extends AbstractSAMLObjectUnmarshaller {

    /** {@inheritDoc} */
    @Override
    protected void processChildElement(@Nonnull final XMLObject parentObject,
            @Nonnull final XMLObject childObject) throws UnmarshallingException {
        final PublicationInfo info = (PublicationInfo) parentObject;

        if (childObject instanceof UsagePolicy) {
            info.getUsagePolicies().add((UsagePolicy)childObject);
        } else {
            super.processChildElement(parentObject, childObject);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void processAttribute(@Nonnull final XMLObject xmlObject, @Nonnull final Attr attribute)
            throws UnmarshallingException {
        final PublicationInfo info = (PublicationInfo) xmlObject;

        if (attribute.getNamespaceURI() == null) {
            if (PublicationInfo.PUBLISHER_ATTRIB_NAME.equals(attribute.getName())) {
                info.setPublisher(attribute.getValue());
            } else if (PublicationInfo.CREATION_INSTANT_ATTRIB_NAME.equals(attribute.getName())) {
                info.setCreationInstant(DOMTypeSupport.stringToInstant(attribute.getValue()));
            } else if (PublicationInfo.PUBLICATION_ID_ATTRIB_NAME.equals(attribute.getName())) {
                info.setPublicationId(attribute.getValue());
            } else {
                super.processAttribute(xmlObject, attribute);
            }
        } else {
            super.processAttribute(xmlObject, attribute);
        }
    }

}