// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;

import org.refcodes.component.Closable;
import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * The {@link ReceiverDecorator} decorates a {@link Provider} with the
 * additional methods of a {@link Receiver} making it easy to use a
 * {@link Provider} wherever a {@link Receiver} is expected.
 * 
 * This is a very plain implementation: {@link #hasDatagram()} always returns
 * true if {@link #isOpened()} is true and as long as none of the
 * {@link #readDatagram()}, {@link #readDatagrams()} or
 * {@link #readDatagrams(int)} methods threw an {@link OpenException} (
 * {@link InterruptedException}).
 * 
 * This implementation actually behaves as if {@link #close()} has been called
 * after verifying {@link #hasDatagram()} and before any of the read methods
 * have been called.
 *
 * @param <DATA> the generic type
 */
public class ReceiverDecorator<DATA extends Serializable> extends AbstractReceiver<DATA> implements Receiver<DATA> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Provider<DATA> _provider;

	private boolean _hasReadSuccess = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new receiver decorator.
	 *
	 * @param aProvider the provider
	 */
	public ReceiverDecorator( Provider<DATA> aProvider ) {
		_provider = aProvider;
		try {
			open();
		}
		catch ( OpenException ignore ) {}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DATA readDatagram() throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _provider.readDatagram();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	synchronized public boolean hasDatagram() throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _hasReadSuccess && isOpened();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DATA[] readDatagrams() throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _provider.readDatagrams();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DATA[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _provider.readDatagrams( aBlockSize );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		if ( _provider instanceof Closable ) {
			((Closable) _provider).close();
		}
		super.close();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
