// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.OpenException;

/**
 * The {@link ByteBlockProvider} is used to receive byte blocks (arrays) in a
 * unified way. The {@link #readDatagrams()} method provides the next available
 * byte block from the counterpart {@link BlockSender} or {@link DatagramSender}
 * ; in case there is none available, then this method halts until one is
 * available.
 */
@FunctionalInterface
public interface ByteBlockProvider {

	/**
	 * Reads (receives) the next byte block passed from a {@link BlockSender} or
	 * {@link DatagramSender} counterpart. In case none byte block is available,
	 * then this method blocks until one is available.
	 * 
	 * When a {@link Thread} is waiting for a byte to be read and
	 * {@link Thread#interrupt()} is being called, then the operation is aborted
	 * and an {@link InterruptedException} is thrown.
	 * 
	 * @return The next byte block sent from the {@link BlockSender} or
	 *         {@link DatagramReceiver} counterpart.
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 * 
	 * @throws InterruptedException Thrown when a {@link Thread} is waiting,
	 *         sleeping, or otherwise occupied, and the {@link Thread} is
	 *         interrupted, either before or during the activity.
	 */
	default byte[] readDatagrams() throws OpenException, InterruptedException {
		return readDatagrams( -1 );
	}

	/**
	 * Similar to {@link #readDatagrams()} though at maximum the amount of data
	 * as provided by the block-size is returned.
	 * 
	 * @param aBlockSize The block-size which is not to exceeded by the returned
	 *        data. A value of -1 specifies to retrieve all available datagrams
	 *        (same behavior as method {@link #readDatagrams()}.
	 * 
	 * @return The next short block sent from the {@link BlockSender} or
	 *         {@link DatagramReceiver} counterpart.
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 * 
	 * @throws InterruptedException Thrown when a {@link Thread} is waiting,
	 *         sleeping, or otherwise occupied, and the {@link Thread} is
	 *         interrupted, either before or during the activity.
	 */
	byte[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException;

}
