// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;

import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * Implementation of the {@link LoopbackSender} interface.
 * 
 * @param <DATA> The type of the datagram to be operated with.
 */
public class LoopbackSenderImpl<DATA extends Serializable> extends AbstractSender<DATA> implements LoopbackSender<DATA> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private LoopbackReceiver<DATA> _loopbackReceiver;

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpenable( LoopbackReceiver<DATA> aLoopbackReceiver ) {
		return !isOpened() && aLoopbackReceiver != null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void open( LoopbackReceiver<DATA> aLoopbackReceiver ) throws OpenException {
		if ( isOpened() ) {
			if ( _loopbackReceiver == aLoopbackReceiver && _loopbackReceiver.isOpened() ) {
				return;
			}
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is " + getConnectionStatus() + "." );
		}
		super.open();
		_loopbackReceiver = aLoopbackReceiver;
		if ( _loopbackReceiver.isOpenable( this ) ) {
			_loopbackReceiver.open( this );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void writeDatagram( DATA aDatagram ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagram.getClass().getName() + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagram( aDatagram );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( DATA[] aDatagrams, int aOffset, int aLength ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagrams + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagrams( aDatagrams, aOffset, aLength );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( DATA[] aDatagrams ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagrams + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagrams( aDatagrams );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		// Nothing to do as datagrams are pushed directly to loopback receiver
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		super.close();
		if ( _loopbackReceiver != null && !_loopbackReceiver.isClosed() ) {
			_loopbackReceiver.close();
			_loopbackReceiver = null;
		}
	}
}
