// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.refcodes.component.OpenException;
import org.refcodes.data.LoopSleepTime;

/**
 * The Interface Receiver.
 *
 * @param <DATA> the generic type
 */
public interface Receiver<DATA extends Serializable> extends Provider<DATA>, DatagramReceiver<DATA>, BlockReceiver<DATA> {

	/**
	 * Read datagrams.
	 *
	 * @return the DAT a[]
	 * @throws OpenException the open exception
	 * @throws InterruptedException the interrupted exception
	 */
	@SuppressWarnings("unchecked")
	@Override
	default DATA[] readDatagrams() throws OpenException, InterruptedException {
		List<DATA> theData = new ArrayList<>();
		while ( hasDatagram() ) {
			theData.add( readDatagram() );
		}
		return theData.toArray( ((DATA[]) new Object[theData.size()]) );
	}

	/**
	 * Read datagrams.
	 *
	 * @param aBlockSize the block size
	 * @return the DAT a[]
	 * @throws OpenException the open exception
	 * @throws InterruptedException the interrupted exception
	 */
	@SuppressWarnings("unchecked")
	@Override
	default DATA[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException {
		DATA[] theBlock = (DATA[]) new Object[aBlockSize];
		int i = 0;
		while ( hasDatagram() && i < aBlockSize ) {
			theBlock[i] = readDatagram();
			i++;
		}
		if ( i == 0 ) {
			while ( !hasDatagram() ) {
				Thread.sleep( LoopSleepTime.NORM.getMilliseconds() );
				while ( hasDatagram() && i < aBlockSize ) {
					theBlock[i] = readDatagram();
					i++;
				}
			}
		}
		if ( i == aBlockSize ) {
			return theBlock;
		}
		return Arrays.copyOfRange( theBlock, 0, i );
	}
}
