// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.Closable;
import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * The {@link ShortReceiverDecorator} decorates a {@link ShortProvider} with the
 * additional methods of a {@link ShortReceiver} making it easy to use a
 * {@link ShortProvider} wherever a {@link ShortReceiver} is expected.
 * 
 * This is a very plain implementation: {@link #hasDatagram()} always returns
 * true if {@link #isOpened()} is true and as long as none of the
 * {@link #readDatagram()}, {@link #readDatagrams()} or
 * {@link #readDatagrams(int)} methods threw an {@link OpenException} (
 * {@link InterruptedException}).
 * 
 * This implementation actually behaves as if {@link #close()} has been called
 * after verifying {@link #hasDatagram()} and before any of the read methods
 * have been called.
 */
public class ShortReceiverDecorator extends AbstractShortReceiver implements ShortReceiver {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private ShortProvider _shortProvider;

	private boolean _hasReadSuccess = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new short receiver decorator.
	 *
	 * @param aShortProvider the short provider
	 */
	public ShortReceiverDecorator( ShortProvider aShortProvider ) {
		_shortProvider = aShortProvider;
		try {
			open();
		}
		catch ( OpenException ignore ) {}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public short readDatagram() throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _shortProvider.readDatagram();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	synchronized public boolean hasDatagram() throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _hasReadSuccess && isOpened();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public short[] readDatagrams() throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _shortProvider.readDatagrams();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public short[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException {
		if ( !isOpened() ) {
			throw new OpenException( "Cannot read as this receiver is in connection status <" + getConnectionStatus() + ">." );
		}
		return _shortProvider.readDatagrams( aBlockSize );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		if ( _shortProvider instanceof Closable ) {
			((Closable) _shortProvider).close();
		}
		super.close();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
