/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.postgresql.connection;

import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.connector.postgresql.PostgresType;
import io.debezium.connector.postgresql.TypeRegistry;
import io.debezium.connector.postgresql.connection.ServerInfo;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.jdbc.JdbcConnection;
import io.debezium.relational.TableId;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.connect.errors.ConnectException;
import org.postgresql.Driver;
import org.postgresql.core.BaseConnection;
import org.postgresql.core.TypeInfo;
import org.postgresql.replication.LogSequenceNumber;
import org.postgresql.util.PSQLState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PostgresConnection
extends JdbcConnection {
    private static final String URL_PATTERN = "jdbc:postgresql://${" + JdbcConfiguration.HOSTNAME + "}:${" + JdbcConfiguration.PORT + "}/${" + JdbcConfiguration.DATABASE + "}";
    protected static JdbcConnection.ConnectionFactory FACTORY = JdbcConnection.patternBasedFactory((String)URL_PATTERN, (String)Driver.class.getName(), (ClassLoader)PostgresConnection.class.getClassLoader(), (Field[])new Field[0]);
    private static Logger LOGGER = LoggerFactory.getLogger(PostgresConnection.class);
    private static final String SQL_NON_ARRAY_TYPES = "SELECT t.oid AS oid, t.typname AS name FROM pg_catalog.pg_type t JOIN pg_catalog.pg_namespace n ON (t.typnamespace = n.oid) WHERE n.nspname != 'pg_toast' AND t.typcategory <> 'A'";
    private static final String SQL_ARRAY_TYPES = "SELECT t.oid AS oid, t.typname AS name, t.typelem AS element FROM pg_catalog.pg_type t JOIN pg_catalog.pg_namespace n ON (t.typnamespace = n.oid) WHERE n.nspname != 'pg_toast' AND t.typcategory = 'A'";
    private final TypeRegistry typeRegistry;

    public PostgresConnection(Configuration config) {
        super(config, FACTORY, PostgresConnection::validateServerVersion, PostgresConnection::defaultSettings);
        try {
            this.typeRegistry = PostgresConnection.initTypeRegistry(this.connection(), this.readTypeInfo());
        }
        catch (SQLException e) {
            throw new ConnectException("Could not intialize type registry", (Throwable)e);
        }
    }

    public String connectionString() {
        return this.connectionString(URL_PATTERN);
    }

    public PostgresConnection executeWithoutCommitting(String ... statements) throws SQLException {
        Connection conn = this.connection();
        try (Statement statement = conn.createStatement();){
            for (String stmt : statements) {
                statement.execute(stmt);
            }
        }
        return this;
    }

    public ServerInfo.ReplicaIdentity readReplicaIdentityInfo(TableId tableId) throws SQLException {
        String statement = "SELECT relreplident FROM pg_catalog.pg_class c LEFT JOIN pg_catalog.pg_namespace n ON c.relnamespace=n.oid WHERE n.nspname=? and c.relname=?";
        String schema = tableId.schema() != null && tableId.schema().length() > 0 ? tableId.schema() : "public";
        StringBuilder replIdentity = new StringBuilder();
        this.prepareQuery(statement, stmt -> {
            stmt.setString(1, schema);
            stmt.setString(2, tableId.table());
        }, rs -> {
            if (rs.next()) {
                replIdentity.append(rs.getString(1));
            } else {
                LOGGER.warn("Cannot determine REPLICA IDENTITY information for table '{}'", (Object)tableId);
            }
        });
        return ServerInfo.ReplicaIdentity.parseFromDB(replIdentity.toString());
    }

    protected ServerInfo.ReplicationSlot readReplicationSlotInfo(String slotName, String pluginName) throws SQLException {
        AtomicReference replicationSlotInfo = new AtomicReference();
        String database = this.database();
        this.prepareQuery("select * from pg_replication_slots where slot_name = ? and database = ? and plugin = ?", statement -> {
            statement.setString(1, slotName);
            statement.setString(2, database);
            statement.setString(3, pluginName);
        }, rs -> {
            if (rs.next()) {
                boolean active = rs.getBoolean("active");
                Long confirmedFlushedLSN = this.parseConfirmedFlushLsn(slotName, pluginName, database, rs);
                replicationSlotInfo.compareAndSet(null, new ServerInfo.ReplicationSlot(active, confirmedFlushedLSN));
            } else {
                LOGGER.debug("No replication slot '{}' is present for plugin '{}' and database '{}'", new Object[]{slotName, pluginName, database});
                replicationSlotInfo.compareAndSet(null, ServerInfo.ReplicationSlot.INVALID);
            }
        });
        return (ServerInfo.ReplicationSlot)replicationSlotInfo.get();
    }

    private Long parseConfirmedFlushLsn(String slotName, String pluginName, String database, ResultSet rs) {
        Long confirmedFlushedLsn = null;
        try {
            String confirmedFlushLSNString = rs.getString("confirmed_flush_lsn");
            if (confirmedFlushLSNString == null) {
                throw new ConnectException("Value confirmed_flush_lsn is missing from the pg_replication_slots table for slot = '" + slotName + "', plugin = '" + pluginName + "', database = '" + database + "'. This is an abnormal situation and the database status should be checked.");
            }
            try {
                confirmedFlushedLsn = LogSequenceNumber.valueOf((String)confirmedFlushLSNString).asLong();
            }
            catch (Exception e) {
                throw new ConnectException("Value confirmed_flush_lsn in the pg_replication_slots table for slot = '" + slotName + "', plugin = '" + pluginName + "', database = '" + database + "' is not valid. This is an abnormal situation and the database status should be checked.");
            }
            if (confirmedFlushedLsn.longValue() == LogSequenceNumber.INVALID_LSN.asLong()) {
                throw new ConnectException("Invalid LSN returned from database");
            }
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
        return confirmedFlushedLsn;
    }

    public boolean dropReplicationSlot(String slotName) {
        try {
            this.execute(new String[]{"select pg_drop_replication_slot('" + slotName + "')"});
            return true;
        }
        catch (SQLException e) {
            PSQLState currentState = new PSQLState(e.getSQLState());
            if (PSQLState.OBJECT_IN_USE.equals((Object)currentState)) {
                LOGGER.warn("Cannot drop replication slot '{}' because it's still in use", (Object)slotName);
                return false;
            }
            if (PSQLState.UNDEFINED_OBJECT.equals((Object)currentState)) {
                LOGGER.debug("Replication slot {} has already been dropped", (Object)slotName);
                return false;
            }
            LOGGER.error("Unexpected error while attempting to drop replication slot", (Throwable)e);
            return false;
        }
    }

    public synchronized void close() {
        try {
            super.close();
        }
        catch (SQLException e) {
            LOGGER.error("Unexpected error while closing Postgres connection", (Throwable)e);
        }
    }

    public Long currentTransactionId() throws SQLException {
        AtomicLong txId = new AtomicLong(0L);
        this.query("select * from txid_current()", rs -> {
            if (rs.next()) {
                txId.compareAndSet(0L, rs.getLong(1));
            }
        });
        long value = txId.get();
        return value > 0L ? Long.valueOf(value) : null;
    }

    public long currentXLogLocation() throws SQLException {
        AtomicLong result = new AtomicLong(0L);
        int majorVersion = this.connection().getMetaData().getDatabaseMajorVersion();
        this.query(majorVersion >= 10 ? "select * from pg_current_wal_lsn()" : "select * from pg_current_xlog_location()", rs -> {
            if (!rs.next()) {
                throw new IllegalStateException("there should always be a valid xlog position");
            }
            result.compareAndSet(0L, LogSequenceNumber.valueOf((String)rs.getString(1)).asLong());
        });
        return result.get();
    }

    public ServerInfo serverInfo() throws SQLException {
        ServerInfo serverInfo = new ServerInfo();
        this.query("SELECT version(), current_user, current_database()", rs -> {
            if (rs.next()) {
                serverInfo.withServer(rs.getString(1)).withUsername(rs.getString(2)).withDatabase(rs.getString(3));
            }
        });
        String username = serverInfo.username();
        if (username != null) {
            this.query("SELECT oid, rolname, rolsuper, rolinherit, rolcreaterole, rolcreatedb, rolcanlogin, rolreplication FROM pg_roles WHERE pg_has_role('" + username + "', oid, 'member')", rs -> {
                while (rs.next()) {
                    String roleInfo = "superuser: " + rs.getBoolean(3) + ", replication: " + rs.getBoolean(8) + ", inherit: " + rs.getBoolean(4) + ", create role: " + rs.getBoolean(5) + ", create db: " + rs.getBoolean(6) + ", can log in: " + rs.getBoolean(7);
                    String roleName = rs.getString(2);
                    serverInfo.addRole(roleName, roleInfo);
                }
            });
        }
        return serverInfo;
    }

    protected static void defaultSettings(Configuration.Builder builder) {
        builder.with("assumeMinServerVersion", "9.4");
    }

    private static void validateServerVersion(Statement statement) throws SQLException {
        DatabaseMetaData metaData = statement.getConnection().getMetaData();
        int majorVersion = metaData.getDatabaseMajorVersion();
        int minorVersion = metaData.getDatabaseMinorVersion();
        if (majorVersion < 9 || majorVersion == 9 && minorVersion < 4) {
            throw new SQLException("Cannot connect to a version of Postgres lower than 9.4");
        }
    }

    protected int resolveNativeType(String typeName) {
        return this.getTypeRegistry().get(typeName).getOid();
    }

    private static TypeRegistry initTypeRegistry(Connection db, Map<String, Integer> nameToJdbc) {
        TypeInfo typeInfo = ((BaseConnection)db).getTypeInfo();
        TypeRegistry.Builder typeRegistryBuilder = TypeRegistry.create(typeInfo);
        try (Statement statement = db.createStatement();){
            try (ResultSet rs = statement.executeQuery(SQL_NON_ARRAY_TYPES);){
                while (rs.next()) {
                    int oid = (int)rs.getLong("oid");
                    typeRegistryBuilder.addType(new PostgresType(rs.getString("name"), oid, nameToJdbc.get(rs.getString("name")), typeInfo));
                }
            }
            rs = statement.executeQuery(SQL_ARRAY_TYPES);
            var7_10 = null;
            try {
                while (rs.next()) {
                    int oid = (int)rs.getLong("oid");
                    typeRegistryBuilder.addType(new PostgresType(rs.getString("name"), oid, nameToJdbc.get(rs.getString("name")), typeInfo, typeRegistryBuilder.get((int)rs.getLong("element"))));
                }
            }
            catch (Throwable throwable) {
                var7_10 = throwable;
                throw throwable;
            }
            finally {
                if (rs != null) {
                    if (var7_10 != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable) {
                            var7_10.addSuppressed(throwable);
                        }
                    } else {
                        rs.close();
                    }
                }
            }
        }
        catch (SQLException e) {
            throw new ConnectException("Could not intialize type registry", (Throwable)e);
        }
        return typeRegistryBuilder.build();
    }

    public TypeRegistry getTypeRegistry() {
        return this.typeRegistry;
    }
}

