package io.fusionauth.security;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.client.OAuth2RestOperations;
import org.springframework.security.oauth2.client.OAuth2RestTemplate;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.common.exceptions.OAuth2Exception;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.web.client.RestTemplate;

/**
 * @author Tyler Scott
 */
public class OpenIDConnectFilter extends AbstractAuthenticationProcessingFilter {
  @Autowired
  private OpenIDAuthorizationCodeResourceDetails openIDResourceDetails;

  private OAuth2RestOperations restTemplate;

  public OpenIDConnectFilter(String defaultFilterProcessesUrl) {
    super(defaultFilterProcessesUrl);
    setAuthenticationManager(new NoopAuthenticationManager());
  }

  @Override
  public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response) throws AuthenticationException {
    OAuth2AccessToken accessToken;
    try {
      accessToken = restTemplate.getAccessToken();
    } catch (final OAuth2Exception e) {
      throw new BadCredentialsException("Could not obtain access token", e);
    }
    try {
      FusionAuthUserDetails user = new FusionAuthUserDetails(getUserInfo(accessToken), accessToken);
      return new UsernamePasswordAuthenticationToken(user, null, user.getAuthorities());
    } catch (Exception e) {
      throw new BadCredentialsException("Failed to validate the token", e);
    }
  }

  public void setRestTemplate(OAuth2RestTemplate restTemplate2) {
    restTemplate = restTemplate2;
  }

  private JsonNode getUserInfo(OAuth2AccessToken accessToken) throws IOException {
    HttpHeaders headers = new HttpHeaders();
    headers.set("Authorization", "Bearer " + accessToken.getValue());

    HttpEntity<String> httpEntity = new HttpEntity<>(headers);
    ResponseEntity<String> response = new RestTemplate().exchange(openIDResourceDetails.getUserInfoUri(), HttpMethod.GET, httpEntity, String.class);
    if (response.getStatusCode() == HttpStatus.OK) {
      return new ObjectMapper().readTree(response.getBody());
    }

    throw new BadCredentialsException("Failed to request user details from the UserInfo API. " +
                                          "Status code [" + response.getStatusCodeValue() + "] Message [" + response.getBody() + "]");
  }

  private static class NoopAuthenticationManager implements AuthenticationManager {
    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
      throw new UnsupportedOperationException("No authentication should be done with this AuthenticationManager");
    }
  }
}
