/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.eventbus;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.reactivex.core.MultiMap;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a message that is received from the event bus in a handler.
 * <p>
 * Messages have a {@link io.vertx.reactivex.core.eventbus.Message#body}, which can be null, and also {@link io.vertx.reactivex.core.eventbus.Message#headers}, which can be empty.
 * <p>
 * If the message was sent specifying a reply handler it will also have a {@link io.vertx.reactivex.core.eventbus.Message#replyAddress}. In that case the message
 * can be replied to using that reply address, or, more simply by just using {@link io.vertx.reactivex.core.eventbus.Message#reply}.
 * <p>
 * If you want to notify the sender that processing failed, then {@link io.vertx.reactivex.core.eventbus.Message#fail} can be called.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.eventbus.Message original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.reactivex.RxGen(io.vertx.core.eventbus.Message.class)
public class Message<T> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    Message that = (Message) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.reactivex.TypeArg<Message> __TYPE_ARG = new io.vertx.lang.reactivex.TypeArg<>(
    obj -> new Message((io.vertx.core.eventbus.Message) obj),
    Message::getDelegate
  );

  private final io.vertx.core.eventbus.Message delegate;
  public final io.vertx.lang.reactivex.TypeArg<T> __typeArg_0;
  
  public Message(io.vertx.core.eventbus.Message delegate) {
    this.delegate = delegate;
    this.__typeArg_0 = io.vertx.lang.reactivex.TypeArg.unknown();
  }

  public Message(io.vertx.core.eventbus.Message delegate, io.vertx.lang.reactivex.TypeArg<T> typeArg_0) {
    this.delegate = delegate;
    this.__typeArg_0 = typeArg_0;
  }

  public io.vertx.core.eventbus.Message getDelegate() {
    return delegate;
  }

  /**
   * The address the message was sent to
   * @return 
   */
  public String address() { 
    String ret = delegate.address();
    return ret;
  }

  /**
   * Multi-map of message headers. Can be empty
   * @return the headers
   */
  public MultiMap headers() { 
    MultiMap ret = MultiMap.newInstance(delegate.headers());
    return ret;
  }

  /**
   * The body of the message. Can be null.
   * @return the body, or null.
   */
  public T body() { 
    if (cached_0 != null) {
      return cached_0;
    }
    T ret = (T)__typeArg_0.wrap(delegate.body());
    cached_0 = ret;
    return ret;
  }

  /**
   * The reply address. Can be null.
   * @return the reply address, or null, if message was sent without a reply handler.
   */
  public String replyAddress() { 
    String ret = delegate.replyAddress();
    return ret;
  }

  /**
   * Signals if this message represents a send or publish event.
   * @return true if this is a send.
   */
  public boolean isSend() { 
    boolean ret = delegate.isSend();
    return ret;
  }

  /**
   * Reply to this message.
   * <p>
   * If the message was sent specifying a reply handler, that handler will be
   * called when it has received a reply. If the message wasn't sent specifying a receipt handler
   * this method does nothing.
   * @param message the message to reply with.
   */
  public void reply(Object message) { 
    delegate.reply(message);
  }

  /**
   * The same as <code>reply(R message)</code> but you can specify handler for the reply - i.e.
   * to receive the reply to the reply.
   * @param message the message to reply with.
   * @param replyHandler the reply handler for the reply.
   */
  public <R> void reply(Object message, Handler<AsyncResult<Message<R>>> replyHandler) { 
    delegate.reply(message, new Handler<AsyncResult<io.vertx.core.eventbus.Message<R>>>() {
      public void handle(AsyncResult<io.vertx.core.eventbus.Message<R>> ar) {
        if (ar.succeeded()) {
          replyHandler.handle(io.vertx.core.Future.succeededFuture(Message.newInstance(ar.result(), io.vertx.lang.reactivex.TypeArg.unknown())));
        } else {
          replyHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * The same as <code>reply(R message)</code> but you can specify handler for the reply - i.e.
   * to receive the reply to the reply.
   * @param message the message to reply with.
   * @return 
   */
  public <R> Single<Message<R>> rxReply(Object message) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<Message<R>>(handler -> {
      reply(message, handler);
    });
  }

  /**
   * Link {@link io.vertx.reactivex.core.eventbus.Message#reply} but allows you to specify delivery options for the reply.
   * @param message the reply message
   * @param options the delivery options
   */
  public void reply(Object message, DeliveryOptions options) { 
    delegate.reply(message, options);
  }

  /**
   * The same as <code>reply(R message, DeliveryOptions)</code> but you can specify handler for the reply - i.e.
   * to receive the reply to the reply.
   * @param message the reply message
   * @param options the delivery options
   * @param replyHandler the reply handler for the reply.
   */
  public <R> void reply(Object message, DeliveryOptions options, Handler<AsyncResult<Message<R>>> replyHandler) { 
    delegate.reply(message, options, new Handler<AsyncResult<io.vertx.core.eventbus.Message<R>>>() {
      public void handle(AsyncResult<io.vertx.core.eventbus.Message<R>> ar) {
        if (ar.succeeded()) {
          replyHandler.handle(io.vertx.core.Future.succeededFuture(Message.newInstance(ar.result(), io.vertx.lang.reactivex.TypeArg.unknown())));
        } else {
          replyHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * The same as <code>reply(R message, DeliveryOptions)</code> but you can specify handler for the reply - i.e.
   * to receive the reply to the reply.
   * @param message the reply message
   * @param options the delivery options
   * @return 
   */
  public <R> Single<Message<R>> rxReply(Object message, DeliveryOptions options) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<Message<R>>(handler -> {
      reply(message, options, handler);
    });
  }

  /**
   * Signal to the sender that processing of this message failed.
   * <p>
   * If the message was sent specifying a result handler
   * the handler will be called with a failure corresponding to the failure code and message specified here.
   * @param failureCode A failure code to pass back to the sender
   * @param message A message to pass back to the sender
   */
  public void fail(int failureCode, String message) { 
    delegate.fail(failureCode, message);
  }

  private T cached_0;

  public static <T>Message<T> newInstance(io.vertx.core.eventbus.Message arg) {
    return arg != null ? new Message<T>(arg) : null;
  }

  public static <T>Message<T> newInstance(io.vertx.core.eventbus.Message arg, io.vertx.lang.reactivex.TypeArg<T> __typeArg_T) {
    return arg != null ? new Message<T>(arg, __typeArg_T) : null;
  }
}
