/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.file;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.reactivex.core.buffer.Buffer;
import io.vertx.reactivex.core.streams.WriteStream;
import io.vertx.reactivex.core.streams.ReadStream;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a file on the file-system which can be read from, or written to asynchronously.
 * <p>
 * This class also implements {@link io.vertx.reactivex.core.streams.ReadStream} and
 * {@link io.vertx.reactivex.core.streams.WriteStream}. This allows the data to be pumped to and from
 * other streams, e.g. an {@link io.vertx.reactivex.core.http.HttpClientRequest} instance,
 * using the {@link io.vertx.reactivex.core.streams.Pump} class
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.file.AsyncFile original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.reactivex.RxGen(io.vertx.core.file.AsyncFile.class)
public class AsyncFile implements ReadStream<Buffer>,  WriteStream<Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    AsyncFile that = (AsyncFile) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.reactivex.TypeArg<AsyncFile> __TYPE_ARG = new io.vertx.lang.reactivex.TypeArg<>(
    obj -> new AsyncFile((io.vertx.core.file.AsyncFile) obj),
    AsyncFile::getDelegate
  );

  private final io.vertx.core.file.AsyncFile delegate;
  
  public AsyncFile(io.vertx.core.file.AsyncFile delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.file.AsyncFile getDelegate() {
    return delegate;
  }

  private io.reactivex.Observable<Buffer> observable;
  private io.reactivex.Flowable<Buffer> flowable;

  public synchronized io.reactivex.Observable<Buffer> toObservable() {
    if (observable == null) {
      java.util.function.Function<io.vertx.core.buffer.Buffer, Buffer> conv = Buffer::newInstance;
      observable = io.vertx.reactivex.ObservableHelper.toObservable(delegate, conv);
    }
    return observable;
  }

  public synchronized io.reactivex.Flowable<Buffer> toFlowable() {
    if (flowable == null) {
      java.util.function.Function<io.vertx.core.buffer.Buffer, Buffer> conv = Buffer::newInstance;
      flowable = io.vertx.reactivex.FlowableHelper.toFlowable(delegate, conv);
    }
    return flowable;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.file.AsyncFile#end} but writes some data to the stream before ending.
   * @param t 
   */
  public void end(Buffer t) { 
    delegate.end(t.getDelegate());
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.file.AsyncFile#setWriteQueueMaxSize}
   * @return true if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public AsyncFile handler(Handler<Buffer> handler) { 
    delegate.handler(new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        handler.handle(Buffer.newInstance(event));
      }
    });
    return this;
  }

  public AsyncFile pause() { 
    delegate.pause();
    return this;
  }

  public AsyncFile resume() { 
    delegate.resume();
    return this;
  }

  public AsyncFile endHandler(Handler<Void> endHandler) { 
    delegate.endHandler(endHandler);
    return this;
  }

  public AsyncFile write(Buffer data) { 
    delegate.write(data.getDelegate());
    return this;
  }

  public AsyncFile setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public AsyncFile drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  public AsyncFile exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  /**
   * Close the file, see {@link io.vertx.reactivex.core.file.AsyncFile#close}.
   */
  public void end() { 
    delegate.end();
  }

  /**
   * Close the file. The actual close happens asynchronously.
   */
  public void close() { 
    delegate.close();
  }

  /**
   * Close the file. The actual close happens asynchronously.
   * The handler will be called when the close is complete, or an error occurs.
   * @param handler the handler
   */
  public void close(Handler<AsyncResult<Void>> handler) { 
    delegate.close(handler);
  }

  /**
   * Close the file. The actual close happens asynchronously.
   * The handler will be called when the close is complete, or an error occurs.
   * @return 
   */
  public Completable rxClose() { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      close(handler);
    });
  }

  /**
   * Write a {@link io.vertx.reactivex.core.buffer.Buffer} to the file at position <code>position</code> in the file, asynchronously.
   * <p>
   * If <code>position</code> lies outside of the current size
   * of the file, the file will be enlarged to encompass it.
   * <p>
   * When multiple writes are invoked on the same file
   * there are no guarantees as to order in which those writes actually occur
   * <p>
   * The handler will be called when the write is complete, or if an error occurs.
   * @param buffer the buffer to write
   * @param position the position in the file to write it at
   * @param handler the handler to call when the write is complete
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile write(Buffer buffer, long position, Handler<AsyncResult<Void>> handler) { 
    delegate.write(buffer.getDelegate(), position, handler);
    return this;
  }

  /**
   * Write a {@link io.vertx.reactivex.core.buffer.Buffer} to the file at position <code>position</code> in the file, asynchronously.
   * <p>
   * If <code>position</code> lies outside of the current size
   * of the file, the file will be enlarged to encompass it.
   * <p>
   * When multiple writes are invoked on the same file
   * there are no guarantees as to order in which those writes actually occur
   * <p>
   * The handler will be called when the write is complete, or if an error occurs.
   * @param buffer the buffer to write
   * @param position the position in the file to write it at
   * @return 
   */
  public Completable rxWrite(Buffer buffer, long position) { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      write(buffer, position, handler);
    });
  }

  /**
   * Reads <code>length</code> bytes of data from the file at position <code>position</code> in the file, asynchronously.
   * <p>
   * The read data will be written into the specified <code>Buffer buffer</code> at position <code>offset</code>.
   * <p>
   * If data is read past the end of the file then zero bytes will be read.<p>
   * When multiple reads are invoked on the same file there are no guarantees as to order in which those reads actually occur.
   * <p>
   * The handler will be called when the close is complete, or if an error occurs.
   * @param buffer the buffer to read into
   * @param offset the offset into the buffer where the data will be read
   * @param position the position in the file where to start reading
   * @param length the number of bytes to read
   * @param handler the handler to call when the write is complete
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile read(Buffer buffer, int offset, long position, int length, Handler<AsyncResult<Buffer>> handler) { 
    delegate.read(buffer.getDelegate(), offset, position, length, new Handler<AsyncResult<io.vertx.core.buffer.Buffer>>() {
      public void handle(AsyncResult<io.vertx.core.buffer.Buffer> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(Buffer.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Reads <code>length</code> bytes of data from the file at position <code>position</code> in the file, asynchronously.
   * <p>
   * The read data will be written into the specified <code>Buffer buffer</code> at position <code>offset</code>.
   * <p>
   * If data is read past the end of the file then zero bytes will be read.<p>
   * When multiple reads are invoked on the same file there are no guarantees as to order in which those reads actually occur.
   * <p>
   * The handler will be called when the close is complete, or if an error occurs.
   * @param buffer the buffer to read into
   * @param offset the offset into the buffer where the data will be read
   * @param position the position in the file where to start reading
   * @param length the number of bytes to read
   * @return 
   */
  public Single<Buffer> rxRead(Buffer buffer, int offset, long position, int length) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<Buffer>(handler -> {
      read(buffer, offset, position, length, handler);
    });
  }

  /**
   * Flush any writes made to this file to underlying persistent storage.
   * <p>
   * If the file was opened with <code>flush</code> set to <code>true</code> then calling this method will have no effect.
   * <p>
   * The actual flush will happen asynchronously.
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile flush() { 
    delegate.flush();
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.file.AsyncFile#flush} but the handler will be called when the flush is complete or if an error occurs
   * @param handler 
   * @return 
   */
  public AsyncFile flush(Handler<AsyncResult<Void>> handler) { 
    delegate.flush(handler);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.file.AsyncFile#flush} but the handler will be called when the flush is complete or if an error occurs
   * @return 
   */
  public Completable rxFlush() { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      flush(handler);
    });
  }

  /**
   * Sets the position from which data will be read from when using the file as a {@link io.vertx.reactivex.core.streams.ReadStream}.
   * @param readPos the position in the file
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile setReadPos(long readPos) { 
    delegate.setReadPos(readPos);
    return this;
  }

  /**
   * Sets the position from which data will be written when using the file as a {@link io.vertx.reactivex.core.streams.WriteStream}.
   * @param writePos the position in the file
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile setWritePos(long writePos) { 
    delegate.setWritePos(writePos);
    return this;
  }

  /**
   * Sets the buffer size that will be used to read the data from the file. Changing this value will impact how much
   * the data will be read at a time from the file system.
   * @param readBufferSize the buffer size
   * @return a reference to this, so the API can be used fluently
   */
  public AsyncFile setReadBufferSize(int readBufferSize) { 
    delegate.setReadBufferSize(readBufferSize);
    return this;
  }


  public static  AsyncFile newInstance(io.vertx.core.file.AsyncFile arg) {
    return arg != null ? new AsyncFile(arg) : null;
  }
}
