/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.reactivex.core.buffer.Buffer;
import io.vertx.reactivex.core.streams.WriteStream;
import io.vertx.core.http.HttpMethod;
import io.vertx.reactivex.core.MultiMap;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a server-side HTTP response.
 * <p>
 * An instance of this is created and associated to every instance of
 * {@link io.vertx.reactivex.core.http.HttpServerRequest} that.
 * <p>
 * It allows the developer to control the HTTP response that is sent back to the
 * client for a particular HTTP request.
 * <p>
 * It contains methods that allow HTTP headers and trailers to be set, and for a body to be written out to the response.
 * <p>
 * It also allows files to be streamed by the kernel directly from disk to the
 * outgoing HTTP connection, bypassing user space altogether (where supported by
 * the underlying operating system). This is a very efficient way of
 * serving files from the server since buffers do not have to be read one by one
 * from the file and written to the outgoing socket.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pump} to pump data with flow control.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpServerResponse original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.reactivex.RxGen(io.vertx.core.http.HttpServerResponse.class)
public class HttpServerResponse implements WriteStream<Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpServerResponse that = (HttpServerResponse) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.reactivex.TypeArg<HttpServerResponse> __TYPE_ARG = new io.vertx.lang.reactivex.TypeArg<>(
    obj -> new HttpServerResponse((io.vertx.core.http.HttpServerResponse) obj),
    HttpServerResponse::getDelegate
  );

  private final io.vertx.core.http.HttpServerResponse delegate;
  
  public HttpServerResponse(io.vertx.core.http.HttpServerResponse delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpServerResponse getDelegate() {
    return delegate;
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpServerResponse#setWriteQueueMaxSize}
   * @return true if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public HttpServerResponse exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public HttpServerResponse write(Buffer data) { 
    delegate.write(data.getDelegate());
    return this;
  }

  public HttpServerResponse setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public HttpServerResponse drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * @return the HTTP status code of the response. The default is <code>200</code> representing <code>OK</code>.
   */
  public int getStatusCode() { 
    int ret = delegate.getStatusCode();
    return ret;
  }

  /**
   * Set the status code. If the status message hasn't been explicitly set, a default status message corresponding
   * to the code will be looked-up and used.
   * @param statusCode 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse setStatusCode(int statusCode) { 
    delegate.setStatusCode(statusCode);
    return this;
  }

  /**
   * @return the HTTP status message of the response. If this is not specified a default value will be used depending on what {@link io.vertx.reactivex.core.http.HttpServerResponse#setStatusCode} has been set to.
   */
  public String getStatusMessage() { 
    String ret = delegate.getStatusMessage();
    return ret;
  }

  /**
   * Set the status message
   * @param statusMessage 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse setStatusMessage(String statusMessage) { 
    delegate.setStatusMessage(statusMessage);
    return this;
  }

  /**
   * If <code>chunked</code> is <code>true</code>, this response will use HTTP chunked encoding, and each call to write to the body
   * will correspond to a new HTTP chunk sent on the wire.
   * <p>
   * If chunked encoding is used the HTTP header <code>Transfer-Encoding</code> with a value of <code>Chunked</code> will be
   * automatically inserted in the response.
   * <p>
   * If <code>chunked</code> is <code>false</code>, this response will not use HTTP chunked encoding, and therefore the total size
   * of any data that is written in the respone body must be set in the <code>Content-Length</code> header <b>before</b> any
   * data is written out.
   * <p>
   * An HTTP chunked response is typically used when you do not know the total size of the request body up front.
   * @param chunked 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return is the response chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    MultiMap ret = MultiMap.newInstance(delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name the header name
   * @param value the header value.
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return The HTTP trailers
   */
  public MultiMap trailers() { 
    if (cached_1 != null) {
      return cached_1;
    }
    MultiMap ret = MultiMap.newInstance(delegate.trailers());
    cached_1 = ret;
    return ret;
  }

  /**
   * Put an HTTP trailer
   * @param name the trailer name
   * @param value the trailer value
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse putTrailer(String name, String value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Set a close handler for the response, this is called when the underlying connection is closed and the response
   * was still using the connection.
   * <p>
   * For HTTP/1.x it is called when the connection is closed before <code>end()</code> is called, therefore it is not
   * guaranteed to be called.
   * <p>
   * For HTTP/2 it is called when the related stream is closed, and therefore it will be always be called.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse closeHandler(Handler<Void> handler) { 
    delegate.closeHandler(handler);
    return this;
  }

  /**
   * Set an end handler for the response. This will be called when the response is disposed to allow consistent cleanup
   * of the response.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse endHandler(Handler<Void> handler) { 
    delegate.endHandler(handler);
    return this;
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded using the encoding <code>enc</code>.
   * @param chunk the string to write
   * @param enc the encoding to use
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse write(String chunk, String enc) { 
    delegate.write(chunk, enc);
    return this;
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded in UTF-8.
   * @param chunk the string to write
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse write(String chunk) { 
    delegate.write(chunk);
    return this;
  }

  /**
   * Used to write an interim 100 Continue response to signify that the client should send the rest of the request.
   * Must only be used if the request contains an "Expect:100-Continue" header
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse writeContinue() { 
    delegate.writeContinue();
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String in UTF-8 encoding before ending the response.
   * @param chunk the string to write before ending the response
   */
  public void end(String chunk) { 
    delegate.end(chunk);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String with the specified encoding before ending the response.
   * @param chunk the string to write before ending the response
   * @param enc the encoding to use
   */
  public void end(String chunk, String enc) { 
    delegate.end(chunk, enc);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes some data to the response body before ending. If the response is not chunked and
   * no other data has been written then the @code{Content-Length} header will be automatically set.
   * @param chunk the buffer to write before ending the response
   */
  public void end(Buffer chunk) { 
    delegate.end(chunk.getDelegate());
  }

  /**
   * Ends the response. If no data has been written to the response body,
   * the actual response won't get written until this method gets called.
   * <p>
   * Once the response has ended, it cannot be used any more.
   */
  public void end() { 
    delegate.end();
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using offset @code{0} which means starting from the beginning of the file.
   * @param filename path to the file to serve
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename) { 
    delegate.sendFile(filename);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename, long offset) { 
    delegate.sendFile(filename, offset);
    return this;
  }

  /**
   * Ask the OS to stream a file as specified by <code>filename</code> directly
   * from disk to the outgoing connection, bypassing userspace altogether
   * (where supported by the underlying operating system.
   * This is a very efficient way to serve files.<p>
   * The actual serve is asynchronous and may not complete until some time after this method has returned.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @param length length to serve to
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename, long offset, long length) { 
    delegate.sendFile(filename, offset, length);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @return 
   */
  public Completable rxSendFile(String filename) { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      sendFile(filename, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename, long offset, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, offset, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @return 
   */
  public Completable rxSendFile(String filename, long offset) { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      sendFile(filename, offset, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length the length to serve to
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse sendFile(String filename, long offset, long length, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, offset, length, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length the length to serve to
   * @return 
   */
  public Completable rxSendFile(String filename, long offset, long length) { 
    return new io.vertx.reactivex.core.impl.AsyncResultCompletable(handler -> {
      sendFile(filename, offset, length, handler);
    });
  }

  /**
   * Close the underlying TCP connection corresponding to the request.
   */
  public void close() { 
    delegate.close();
  }

  /**
   * @return has the response already ended?
   */
  public boolean ended() { 
    boolean ret = delegate.ended();
    return ret;
  }

  /**
   * @return has the underlying TCP connection corresponding to the request already been closed?
   */
  public boolean closed() { 
    boolean ret = delegate.closed();
    return ret;
  }

  /**
   * @return have the headers for the response already been written?
   */
  public boolean headWritten() { 
    boolean ret = delegate.headWritten();
    return ret;
  }

  /**
   * Provide a handler that will be called just before the headers are written to the wire.<p>
   * This provides a hook allowing you to add any more headers or do any more operations before this occurs.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse headersEndHandler(Handler<Void> handler) { 
    delegate.headersEndHandler(handler);
    return this;
  }

  /**
   * Provides a handler that will be called after the last part of the body is written to the wire.
   * The handler is called asynchronously of when the response has been received by the client.
   * This provides a hook allowing you to do more operations once the request has been sent over the wire
   * such as resource cleanup.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse bodyEndHandler(Handler<Void> handler) { 
    delegate.bodyEndHandler(handler);
    return this;
  }

  /**
   * @return the total number of bytes written for the body of the response.
   */
  public long bytesWritten() { 
    long ret = delegate.bytesWritten();
    return ret;
  }

  /**
   * @return the id of the stream of this response,  for HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @param handler 
   * @return 
   */
  public HttpServerResponse push(HttpMethod method, String host, String path, Handler<AsyncResult<HttpServerResponse>> handler) { 
    HttpServerResponse ret = HttpServerResponse.newInstance(delegate.push(method, host, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @param handler 
   * @return 
   */
  public HttpServerResponse push(HttpMethod method, String path, MultiMap headers, Handler<AsyncResult<HttpServerResponse>> handler) { 
    HttpServerResponse ret = HttpServerResponse.newInstance(delegate.push(method, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    }));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param handler 
   * @return 
   */
  public HttpServerResponse push(HttpMethod method, String path, Handler<AsyncResult<HttpServerResponse>> handler) { 
    delegate.push(method, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public Single<HttpServerResponse> rxPush(HttpMethod method, String path) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpServerResponse>(handler -> {
      push(method, path, handler);
    });
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @param handler the handler notified when the response can be written
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse push(HttpMethod method, String host, String path, MultiMap headers, Handler<AsyncResult<HttpServerResponse>> handler) { 
    delegate.push(method, host, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return 
   */
  public Single<HttpServerResponse> rxPush(HttpMethod method, String host, String path, MultiMap headers) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpServerResponse>(handler -> {
      push(method, host, path, headers, handler);
    });
  }

  /**
   * Reset this HTTP/2 stream with the error code <code>0</code>.
   */
  public void reset() { 
    delegate.reset();
  }

  /**
   * Reset this HTTP/2 stream with the error <code>code</code>.
   * @param code the error code
   */
  public void reset(long code) { 
    delegate.reset(code);
  }

  /**
   * Write an HTTP/2 frame to the response, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public HttpServerResponse writeCustomFrame(int type, int flags, Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#writeCustomFrame} but with an {@link io.vertx.reactivex.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public HttpServerResponse writeCustomFrame(HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  private MultiMap cached_0;
  private MultiMap cached_1;

  public static  HttpServerResponse newInstance(io.vertx.core.http.HttpServerResponse arg) {
    return arg != null ? new HttpServerResponse(arg) : null;
  }
}
