/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferRecycler;
import org.apache.flink.runtime.io.network.buffer.NetworkBuffer;
import org.apache.flink.runtime.io.network.partition.BufferWithChannel;
import org.apache.flink.runtime.io.network.partition.DataBuffer;
import org.apache.flink.util.Preconditions;

@NotThreadSafe
public class SortBasedDataBuffer
implements DataBuffer {
    private static final int INDEX_ENTRY_SIZE = 16;
    private final LinkedList<MemorySegment> freeSegments;
    private final BufferRecycler bufferRecycler;
    private final ArrayList<MemorySegment> segments = new ArrayList();
    private final long[] firstIndexEntryAddresses;
    private final long[] lastIndexEntryAddresses;
    private final int bufferSize;
    private final int numGuaranteedBuffers;
    private long numTotalBytes;
    private long numTotalRecords;
    private long numTotalBytesRead;
    private boolean isFinished;
    private boolean isReleased;
    private int writeSegmentIndex;
    private int writeSegmentOffset;
    private final int[] subpartitionReadOrder;
    private long readIndexEntryAddress;
    private int recordRemainingBytes;
    private int readOrderIndex = -1;

    public SortBasedDataBuffer(LinkedList<MemorySegment> freeSegments, BufferRecycler bufferRecycler, int numSubpartitions, int bufferSize, int numGuaranteedBuffers, @Nullable int[] customReadOrder) {
        Preconditions.checkArgument((bufferSize > 16 ? 1 : 0) != 0, (Object)"Buffer size is too small.");
        Preconditions.checkArgument((numGuaranteedBuffers > 0 ? 1 : 0) != 0, (Object)"No guaranteed buffers for sort.");
        this.freeSegments = (LinkedList)Preconditions.checkNotNull(freeSegments);
        this.bufferRecycler = (BufferRecycler)Preconditions.checkNotNull((Object)bufferRecycler);
        this.bufferSize = bufferSize;
        this.numGuaranteedBuffers = numGuaranteedBuffers;
        Preconditions.checkState((numGuaranteedBuffers <= freeSegments.size() ? 1 : 0) != 0, (Object)"Wrong number of free segments.");
        this.firstIndexEntryAddresses = new long[numSubpartitions];
        this.lastIndexEntryAddresses = new long[numSubpartitions];
        Arrays.fill(this.firstIndexEntryAddresses, -1L);
        Arrays.fill(this.lastIndexEntryAddresses, -1L);
        this.subpartitionReadOrder = new int[numSubpartitions];
        if (customReadOrder != null) {
            Preconditions.checkArgument((customReadOrder.length == numSubpartitions ? 1 : 0) != 0, (Object)"Illegal data read order.");
            System.arraycopy(customReadOrder, 0, this.subpartitionReadOrder, 0, numSubpartitions);
        } else {
            for (int channel = 0; channel < numSubpartitions; ++channel) {
                this.subpartitionReadOrder[channel] = channel;
            }
        }
    }

    @Override
    public boolean append(ByteBuffer source, int targetChannel, Buffer.DataType dataType) throws IOException {
        Preconditions.checkArgument((boolean)source.hasRemaining(), (Object)"Cannot append empty data.");
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"Sort buffer is already finished.");
        Preconditions.checkState((!this.isReleased ? 1 : 0) != 0, (Object)"Sort buffer is already released.");
        int totalBytes = source.remaining();
        if (!this.allocateBuffersForRecord(totalBytes)) {
            return true;
        }
        this.writeIndex(targetChannel, totalBytes, dataType);
        this.writeRecord(source);
        ++this.numTotalRecords;
        this.numTotalBytes += (long)totalBytes;
        return false;
    }

    private void writeIndex(int channelIndex, int numRecordBytes, Buffer.DataType dataType) {
        MemorySegment segment = this.segments.get(this.writeSegmentIndex);
        segment.putLong(this.writeSegmentOffset, (long)numRecordBytes << 32 | (long)dataType.ordinal());
        long indexEntryAddress = (long)this.writeSegmentIndex << 32 | (long)this.writeSegmentOffset;
        long lastIndexEntryAddress = this.lastIndexEntryAddresses[channelIndex];
        this.lastIndexEntryAddresses[channelIndex] = indexEntryAddress;
        if (lastIndexEntryAddress >= 0L) {
            segment = this.segments.get(this.getSegmentIndexFromPointer(lastIndexEntryAddress));
            segment.putLong(this.getSegmentOffsetFromPointer(lastIndexEntryAddress) + 8, indexEntryAddress);
        } else {
            this.firstIndexEntryAddresses[channelIndex] = indexEntryAddress;
        }
        this.updateWriteSegmentIndexAndOffset(16);
    }

    private void writeRecord(ByteBuffer source) {
        while (source.hasRemaining()) {
            MemorySegment segment = this.segments.get(this.writeSegmentIndex);
            int toCopy = Math.min(this.bufferSize - this.writeSegmentOffset, source.remaining());
            segment.put(this.writeSegmentOffset, source, toCopy);
            this.updateWriteSegmentIndexAndOffset(toCopy);
        }
    }

    private boolean allocateBuffersForRecord(int numRecordBytes) {
        int availableBytes;
        int numBytesRequired = 16 + numRecordBytes;
        int n = availableBytes = this.writeSegmentIndex == this.segments.size() ? 0 : this.bufferSize - this.writeSegmentOffset;
        if (availableBytes >= numBytesRequired) {
            return true;
        }
        if (availableBytes < 16) {
            this.updateWriteSegmentIndexAndOffset(availableBytes);
            availableBytes = 0;
        }
        if ((long)availableBytes + (long)(this.numGuaranteedBuffers - this.segments.size()) * (long)this.bufferSize < (long)numBytesRequired) {
            return false;
        }
        do {
            MemorySegment segment = this.freeSegments.poll();
            this.addBuffer((MemorySegment)Preconditions.checkNotNull((Object)segment));
        } while ((availableBytes += this.bufferSize) < numBytesRequired);
        return true;
    }

    private void addBuffer(MemorySegment segment) {
        if (segment.size() != this.bufferSize) {
            this.bufferRecycler.recycle(segment);
            throw new IllegalStateException("Illegal memory segment size.");
        }
        if (this.isReleased) {
            this.bufferRecycler.recycle(segment);
            throw new IllegalStateException("Sort buffer is already released.");
        }
        this.segments.add(segment);
    }

    private void updateWriteSegmentIndexAndOffset(int numBytes) {
        this.writeSegmentOffset += numBytes;
        if (this.writeSegmentOffset == this.bufferSize) {
            ++this.writeSegmentIndex;
            this.writeSegmentOffset = 0;
        }
    }

    @Override
    public BufferWithChannel getNextBuffer(MemorySegment transitBuffer) {
        Preconditions.checkState((boolean)this.isFinished, (Object)"Sort buffer is not ready to be read.");
        Preconditions.checkState((!this.isReleased ? 1 : 0) != 0, (Object)"Sort buffer is already released.");
        if (!this.hasRemaining()) {
            return null;
        }
        int numBytesCopied = 0;
        Buffer.DataType bufferDataType = Buffer.DataType.DATA_BUFFER;
        int channelIndex = this.subpartitionReadOrder[this.readOrderIndex];
        do {
            int sourceSegmentIndex = this.getSegmentIndexFromPointer(this.readIndexEntryAddress);
            int sourceSegmentOffset = this.getSegmentOffsetFromPointer(this.readIndexEntryAddress);
            MemorySegment sourceSegment = this.segments.get(sourceSegmentIndex);
            long lengthAndDataType = sourceSegment.getLong(sourceSegmentOffset);
            int length = this.getSegmentIndexFromPointer(lengthAndDataType);
            Buffer.DataType dataType = Buffer.DataType.values()[this.getSegmentOffsetFromPointer(lengthAndDataType)];
            if (dataType.isEvent() && numBytesCopied > 0) break;
            bufferDataType = dataType;
            long nextReadIndexEntryAddress = sourceSegment.getLong(sourceSegmentOffset + 8);
            sourceSegmentOffset += 16;
            if (bufferDataType.isEvent() && transitBuffer.size() < length) {
                transitBuffer = MemorySegmentFactory.allocateUnpooledSegment((int)length);
            }
            numBytesCopied += this.copyRecordOrEvent(transitBuffer, numBytesCopied, sourceSegmentIndex, sourceSegmentOffset, length);
            if (this.recordRemainingBytes != 0) continue;
            if (this.readIndexEntryAddress == this.lastIndexEntryAddresses[channelIndex]) {
                this.updateReadChannelAndIndexEntryAddress();
                break;
            }
            this.readIndexEntryAddress = nextReadIndexEntryAddress;
        } while (numBytesCopied < transitBuffer.size() && bufferDataType.isBuffer());
        this.numTotalBytesRead += (long)numBytesCopied;
        NetworkBuffer buffer = new NetworkBuffer(transitBuffer, buf -> {}, bufferDataType, numBytesCopied);
        return new BufferWithChannel(buffer, channelIndex);
    }

    private int copyRecordOrEvent(MemorySegment targetSegment, int targetSegmentOffset, int sourceSegmentIndex, int sourceSegmentOffset, int recordLength) {
        int numBytes;
        if (this.recordRemainingBytes > 0) {
            long position = (long)sourceSegmentOffset + (long)(recordLength - this.recordRemainingBytes);
            sourceSegmentIndex = (int)((long)sourceSegmentIndex + position / (long)this.bufferSize);
            sourceSegmentOffset = (int)(position % (long)this.bufferSize);
        } else {
            this.recordRemainingBytes = recordLength;
        }
        int targetSegmentSize = targetSegment.size();
        int numBytesToCopy = Math.min(targetSegmentSize - targetSegmentOffset, this.recordRemainingBytes);
        do {
            if (sourceSegmentOffset == this.bufferSize) {
                ++sourceSegmentIndex;
                sourceSegmentOffset = 0;
            }
            int sourceRemainingBytes = Math.min(this.bufferSize - sourceSegmentOffset, this.recordRemainingBytes);
            numBytes = Math.min(targetSegmentSize - targetSegmentOffset, sourceRemainingBytes);
            MemorySegment sourceSegment = this.segments.get(sourceSegmentIndex);
            sourceSegment.copyTo(sourceSegmentOffset, targetSegment, targetSegmentOffset, numBytes);
            this.recordRemainingBytes -= numBytes;
            sourceSegmentOffset += numBytes;
        } while (this.recordRemainingBytes > 0 && (targetSegmentOffset += numBytes) < targetSegmentSize);
        return numBytesToCopy;
    }

    private void updateReadChannelAndIndexEntryAddress() {
        int channelIndex;
        while (++this.readOrderIndex < this.firstIndexEntryAddresses.length && (this.readIndexEntryAddress = this.firstIndexEntryAddresses[channelIndex = this.subpartitionReadOrder[this.readOrderIndex]]) < 0L) {
        }
    }

    private int getSegmentIndexFromPointer(long value) {
        return (int)(value >>> 32);
    }

    private int getSegmentOffsetFromPointer(long value) {
        return (int)value;
    }

    @Override
    public long numTotalRecords() {
        return this.numTotalRecords;
    }

    @Override
    public long numTotalBytes() {
        return this.numTotalBytes;
    }

    @Override
    public boolean hasRemaining() {
        return this.numTotalBytesRead < this.numTotalBytes;
    }

    @Override
    public void finish() {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"DataBuffer is already finished.");
        this.isFinished = true;
        this.updateReadChannelAndIndexEntryAddress();
    }

    @Override
    public boolean isFinished() {
        return this.isFinished;
    }

    @Override
    public void release() {
        if (this.isReleased) {
            return;
        }
        this.isReleased = true;
        for (MemorySegment segment : this.segments) {
            this.bufferRecycler.recycle(segment);
        }
        this.segments.clear();
    }

    @Override
    public boolean isReleased() {
        return this.isReleased;
    }
}

