/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.api;

import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import org.sonar.squid.measures.Measurable;
import org.sonar.squid.measures.Measures;
import org.sonar.squid.measures.Metric;

public abstract class SourceCode implements Measurable, Comparable<SourceCode> {

  private final String name;
  private Measures measures = new Measures();
  private final String key;
  private int startAtLine = -1;
  private int endAtLine = -1;
  private SourceCode parent;
  private SortedSet<SourceCode> children;
  private SourceCodeIndexer indexer;

  public SourceCode(String key) {
    this(key, null);
  }

  public SourceCode(String key, String name) {
    this.key = key;
    this.name = name;
  }

  public String getKey() {
    return key;
  }

  public int compareTo(SourceCode resource) {
    return key.compareTo(resource.getKey());
  }

  public String getName() {
    return name;
  }

  public final void setSourceCodeIndexer(SourceCodeIndexer indexer) {
    this.indexer = indexer;
  }

  private void index(SourceCode sourceCode) {
    if (indexer != null) {
      indexer.index(sourceCode);
    }
  }

  @Override
  public boolean equals(Object obj) {
    return (obj instanceof SourceCode) && key.equals(((SourceCode) obj).key);
  }

  @Override
  public int hashCode() {
    return key.hashCode();
  }

  @Override
  public String toString() {
    return getKey();
  }

  public final void computeMeasures() {
    if (hasChildren()) {
      for (SourceCode child : getChildren()) {
        child.computeMeasures();
      }
    }
    for (Metric metric : Metric.values()) {
      if (!metric.aggregateIfThereIsAlreadyAValue() && getDouble(metric) != 0) {
        continue;
      }
      if (hasChildren()) {
        for (SourceCode child : getChildren()) {
          if (!metric.isCalculatedMetric() && metric.isThereAggregationFormula()) {
            add(metric, child);
          }
        }
      }
    }
  }

  public boolean isType(Class<? extends SourceCode> resourceType) {
    return this.getClass() == resourceType;
  }

  /**
   * {@inheritDoc}
   */
  public int getInt(Metric metric) {
    return (int) getMeasure(metric);
  }

  /**
   * {@inheritDoc}
   */
  public double getDouble(Metric metric) {
    return getMeasure(metric);
  }

  public void add(Metric metric, SourceCode child) {
    add(metric, child.getMeasure(metric));
  }

  public void add(Metric metric, double value) {
    setMeasure(metric, getMeasure(metric) + value);
  }

  public void addData(Metric metric, Object data) {
    measures.setData(metric, data);
  }

  public Object getData(Metric metric) {
    return measures.getData(metric);
  }

  private double getMeasure(Metric metric) {
    if (metric.isCalculatedMetric()) {
      return metric.getCalculatedMetricFormula().calculate(this);
    }
    return measures.getValue(metric);
  }

  /**
   * {@inheritDoc}
   */
  public void setMeasure(Metric metric, double measure) {
    if (metric.isCalculatedMetric()) {
      throw new IllegalStateException("It's not allowed to set the value of a calculated metric : " + metric.name());
    }
    measures.setValue(metric, measure);
  }

  /**
   * {@inheritDoc}
   */
  public void setMeasure(Metric metric, int measure) {
    setMeasure(metric, (double) measure);
  }

  public void removeMeasure(Metric metric) {
    measures.removeMeasure(metric);
  }

  public void setStartAtLine(int startAtLine) {
    this.startAtLine = startAtLine;
    this.endAtLine = startAtLine;
  }

  public void setEndAtLine(int endAtLine) {
    this.endAtLine = endAtLine;
  }

  public int getStartAtLine() {
    return startAtLine;
  }

  public int getEndAtLine() {
    return endAtLine;
  }

  public SourceCode addChild(SourceCode sourceCode) {
    if (children == null) {
      children = new TreeSet<SourceCode>();
    }
    sourceCode.setParent(this);
    if (!children.contains(sourceCode)) {
      children.add(sourceCode);
      index(sourceCode);
    }
    return this;
  }

  public SourceCode getParent(Class<? extends SourceCode> sourceCode) {
    if (parent == null) {
      return null;
    } else if (parent.getClass().equals(sourceCode)) {
      return parent;
    } else {
      return parent.getParent(sourceCode);
    }
  }

  public SourceCode getFirstChild() {
    return !children.isEmpty() ? children.first() : null;
  }

  public SourceCode getLastChild() {
    return !children.isEmpty() ? children.last() : null;
  }

  private void setParent(SourceCode parent) {
    this.parent = parent;
  }

  public SourceCode getParent() {
    return parent;
  }

  public Set<SourceCode> getChildren() {
    return children;
  }

  public boolean hasChild(SourceCode squidUnit) {
    if (!hasChildren()) {
      return false;
    }
    if (children.contains(squidUnit)) {
      return true;
    }
    for (SourceCode child : children) {
      if (child.hasChild(squidUnit)) {
        return true;
      }
    }
    return false;
  }

  private boolean hasChildren() {
    return children != null && children.size() != 0;
  }

  public boolean hasAmongParents(SourceCode expectedParent) {
    if (parent == null) {
      return false;
    }
    return parent.equals(expectedParent) || parent.hasAmongParents(expectedParent);
  }
}
