/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.percolator;

import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.client.node.NodeClient;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.VersionType;
import org.elasticsearch.rest.BaseRestHandler;
import org.elasticsearch.rest.RestChannel;
import org.elasticsearch.rest.RestController;
import org.elasticsearch.rest.RestRequest;
import org.elasticsearch.rest.action.RestActions;
import org.elasticsearch.rest.action.RestToXContentListener;

import java.io.IOException;

import static org.elasticsearch.rest.RestRequest.Method.GET;
import static org.elasticsearch.rest.RestRequest.Method.POST;

@Deprecated
public class RestPercolateAction extends BaseRestHandler {

    @Inject
    public RestPercolateAction(Settings settings, RestController controller) {
        super(settings);
        controller.registerHandler(GET, "/{index}/{type}/_percolate", this);
        controller.registerHandler(POST, "/{index}/{type}/_percolate", this);

        RestPercolateExistingDocHandler existingDocHandler = new RestPercolateExistingDocHandler(settings);
        controller.registerHandler(GET, "/{index}/{type}/{id}/_percolate", existingDocHandler);
        controller.registerHandler(POST, "/{index}/{type}/{id}/_percolate", existingDocHandler);

        RestCountPercolateDocHandler countHandler = new RestCountPercolateDocHandler(settings);
        controller.registerHandler(GET, "/{index}/{type}/_percolate/count", countHandler);
        controller.registerHandler(POST, "/{index}/{type}/_percolate/count", countHandler);

        RestCountPercolateExistingDocHandler countExistingDocHandler = new RestCountPercolateExistingDocHandler(settings);
        controller.registerHandler(GET, "/{index}/{type}/{id}/_percolate/count", countExistingDocHandler);
        controller.registerHandler(POST, "/{index}/{type}/{id}/_percolate/count", countExistingDocHandler);
    }

    private RestChannelConsumer parseDocPercolate(PercolateRequest percolateRequest, RestRequest restRequest, NodeClient client) {
        percolateRequest.indices(Strings.splitStringByCommaToArray(restRequest.param("index")));
        percolateRequest.documentType(restRequest.param("type"));
        percolateRequest.routing(restRequest.param("routing"));
        percolateRequest.preference(restRequest.param("preference"));
        percolateRequest.source(RestActions.getRestContent(restRequest));

        percolateRequest.indicesOptions(IndicesOptions.fromRequest(restRequest, percolateRequest.indicesOptions()));
        return channel -> executePercolate(client, percolateRequest, channel);
    }

    private RestChannelConsumer parseExistingDocPercolate(PercolateRequest percolateRequest, RestRequest restRequest, NodeClient client) {
        String index = restRequest.param("index");
        String type = restRequest.param("type");
        percolateRequest.indices(Strings.splitStringByCommaToArray(restRequest.param("percolate_index", index)));
        percolateRequest.documentType(restRequest.param("percolate_type", type));

        GetRequest getRequest = new GetRequest(index, type,
                restRequest.param("id"));
        getRequest.routing(restRequest.param("routing"));
        getRequest.preference(restRequest.param("preference"));
        getRequest.refresh(restRequest.paramAsBoolean("refresh", getRequest.refresh()));
        getRequest.realtime(restRequest.paramAsBoolean("realtime", getRequest.realtime()));
        getRequest.version(RestActions.parseVersion(restRequest));
        getRequest.versionType(VersionType.fromString(restRequest.param("version_type"), getRequest.versionType()));

        percolateRequest.getRequest(getRequest);
        percolateRequest.routing(restRequest.param("percolate_routing"));
        percolateRequest.preference(restRequest.param("percolate_preference"));
        percolateRequest.source(RestActions.getRestContent(restRequest));

        percolateRequest.indicesOptions(IndicesOptions.fromRequest(restRequest, percolateRequest.indicesOptions()));
        return channel -> executePercolate(client, percolateRequest, channel);
    }

    private void executePercolate(final NodeClient client, final PercolateRequest percolateRequest, final RestChannel restChannel) {
        client.execute(PercolateAction.INSTANCE, percolateRequest, new RestToXContentListener<>(restChannel));
    }

    @Override
    public RestChannelConsumer prepareRequest(RestRequest restRequest, final NodeClient client) throws IOException {
        PercolateRequest percolateRequest = new PercolateRequest();
        return parseDocPercolate(percolateRequest, restRequest, client);
    }

    private final class RestCountPercolateDocHandler extends BaseRestHandler {

        private RestCountPercolateDocHandler(Settings settings) {
            super(settings);
        }

        @Override
        public RestChannelConsumer prepareRequest(RestRequest restRequest, final NodeClient client) throws IOException {
            PercolateRequest percolateRequest = new PercolateRequest();
            percolateRequest.onlyCount(true);
            return parseDocPercolate(percolateRequest, restRequest, client);
        }

    }

    private final class RestPercolateExistingDocHandler extends BaseRestHandler {

        RestPercolateExistingDocHandler(Settings settings) {
            super(settings);
        }

        @Override
        public RestChannelConsumer prepareRequest(RestRequest restRequest, final NodeClient client) throws IOException {
            PercolateRequest percolateRequest = new PercolateRequest();
            return parseExistingDocPercolate(percolateRequest, restRequest, client);
        }

    }

    private final class RestCountPercolateExistingDocHandler extends BaseRestHandler {

        RestCountPercolateExistingDocHandler(Settings settings) {
            super(settings);
        }

        @Override
        public RestChannelConsumer prepareRequest(RestRequest restRequest, final NodeClient client) throws IOException {
            PercolateRequest percolateRequest = new PercolateRequest();
            percolateRequest.onlyCount(true);
            return parseExistingDocPercolate(percolateRequest, restRequest, client);
        }

    }

}
